import { Injectable } from '@nestjs/common';
import { Blacklist, HamiData } from '../entities/blacklist.entity';
import { InjectDataSource } from '@nestjs/typeorm';
import { DataSource } from 'typeorm';
import { ConfigService } from '@nestjs/config';
import moment from 'moment';
import _ from 'lodash';
import { CrudService, Inner } from 'nicot';

@Injectable()
export class BlacklistService extends CrudService(Blacklist, {
  relations: [Inner('accounts')],
}) {
  constructor(
    @InjectDataSource() db: DataSource,
    private config: ConfigService,
  ) {
    super(db.getRepository(Blacklist));
  }

  async getSinceTime() {
    const [lastEntry] = await this.repo.find({
      take: 1,
      select: { time: true },
      order: { time: 'desc' },
    });
    if (!lastEntry) {
      return this.config.get<string>('INITIAL_TIME') || '2022-01-01 00:00:00';
    }
    return moment(lastEntry.time)
      .add(1, 'second')
      .format('YYYY-MM-DD HH:mm:ss');
  }

  async saveEntries(entries: HamiData[]) {
    this.log.log(`Saving ${entries.length} entries...`);
    const chunks = _.chunk(entries.reverse(), 4000);
    for (const chunk of chunks) {
      const blacklists = chunk.map((d) => new Blacklist().fromData(d));
      this.log.log(`Saving current chunk of ${blacklists.length} entries...`);
      try {
        const result = await this._batchCreate(blacklists);
        this.log.log(
          `Saved ${result.results.length} entries, and skipped ${result.skipped.length} duplicated entries.`,
        );
      } catch (e) {
        this.log.log(`Failed to save ${chunk.length} entries: ${e.message}`);
      }
    }
    this.log.log(`Finished saving ${entries.length} entries.`);
  }
}
