import { Module } from '@nestjs/common';
import { ConfigModule, ConfigService } from '@nestjs/config';
import { loadConfig } from './utility/config';
import { TypeOrmModule } from '@nestjs/typeorm';
import { Blacklist } from './entities/blacklist.entity';
import { BlacklistAccount } from './entities/blacklist-account.entity';
import { BlacklistAccountService } from './blacklist-account/blacklist-account.service';
import { BlacklistAccountController } from './blacklist-account/blacklist-account.controller';
import { BlacklistService } from './blacklist/blacklist.service';
import { FetchService } from './fetch/fetch.service';
import { HttpModule } from '@nestjs/axios';
import { AdService } from './ad/ad.service';
import { AdController } from './ad/ad.controller';
import { Ad } from './entities/ad.entity';
import { WhitelistAccountService } from './whitelist-account/whitelist-account.service';
import { WhitelistAccountController } from './whitelist-account/whitelist-account.controller';
import { WhitelistAccount } from './entities/whitelist-account.entity';

@Module({
  imports: [
    ConfigModule.forRoot({
      load: [loadConfig],
      isGlobal: true,
      ignoreEnvVars: true,
      ignoreEnvFile: true,
    }),
    HttpModule.registerAsync({
      inject: [ConfigService],
      useFactory: async (config: ConfigService) => config.get('http') || {},
    }),
    TypeOrmModule.forFeature([
      Blacklist,
      BlacklistAccount,
      Ad,
      WhitelistAccount,
    ]),
    TypeOrmModule.forRootAsync({
      inject: [ConfigService],
      useFactory: async (config: ConfigService) => ({
        type: 'postgres',
        entities: [],
        autoLoadEntities: true,
        synchronize: !config.get('DB_NO_INIT') || !!config.get('REFETCH'),
        dropSchema: !!config.get('REFETCH'),
        host: config.get('DB_HOST'),
        port: parseInt(config.get('DB_PORT')) || 5432,
        username: config.get('DB_USER'),
        password: config.get('DB_PASS'),
        database: config.get('DB_NAME'),
        supportBigNumbers: true,
        bigNumberStrings: false,
      }),
    }),
  ],
  providers: [
    BlacklistAccountService,
    BlacklistService,
    FetchService,
    AdService,
    WhitelistAccountService,
  ],
  controllers: [
    BlacklistAccountController,
    AdController,
    WhitelistAccountController,
  ],
})
export class AppModule {}
