import { Column, Entity, Index, SelectQueryBuilder } from 'typeorm';
import { IdBase } from './bases/IdBase.entity';
import { StringColumn } from './decorators/base';
import { IsDate, ValidateIf } from 'class-validator';
import { ApiProperty } from '@nestjs/swagger';
import { Session } from 'koishi';
import { applyQueryProperty, applyQueryPropertySearch } from './utility/query';

@Entity()
export class RankRecord extends IdBase {
  @Index()
  @StringColumn(11, 'Operator ID', undefined)
  operatorId: string;

  @Index()
  @StringColumn(11, 'User ID', undefined, true)
  userId: string;

  @Index()
  @StringColumn(11, 'Guild ID', undefined, true)
  guildId: string;

  @Index()
  @Column({ type: 'timestamp' })
  @IsDate()
  @ApiProperty({ type: Date, description: 'The date of the rank given' })
  rankDate: Date;

  @Index()
  @ValidateIf((v) => v?.length)
  @StringColumn(6, 'Rank content', undefined, true)
  rankName: string;

  fromSession(session: Session, targetId?: string) {
    if (targetId) {
      this.userId = targetId;
      this.operatorId = session.userId;
    } else {
      this.userId = session.userId;
      this.operatorId = null;
    }
    this.userId = session.userId;
    this.guildId = session.guildId;
    this.rankDate = new Date();
    return this;
  }

  override applyQuery(qb: SelectQueryBuilder<RankRecord>, entityName: string) {
    super.applyQuery(qb, entityName);
    applyQueryProperty(this, qb, entityName, 'userId', 'guildId', 'rankDate');
    applyQueryPropertySearch(this, qb, entityName, 'rankName');
  }
}
