import { ConsoleLogger, Injectable } from '@nestjs/common';
import {
  OnChannel,
  PutArg,
  PutOption,
  PutSession,
  UseCommand,
} from 'koishi-nestjs';
import { Bot, segment, Session } from 'koishi';
import { RecordService } from '../record/record.service';
import { OneBotBot } from '@koishijs/plugin-adapter-onebot';

@Injectable()
export class CommandService extends ConsoleLogger {
  constructor(private record: RecordService) {
    super('CommandService');
  }

  private async getMember(session: Session, id: string) {
    try {
      const member = await session.bot.getGuildMember(session.guildId, id);
      return member;
    } catch (e) {
      this.warn(`Member ${id} not found.`);
    }
    return;
  }

  private parseTarget(id: string) {
    if (!id) return;
    if (id.match(/^\d+$/)) return id;
    const seg = segment.parse(id);
    const target = seg.find((s) => s.type === 'at');
    if (!target) return;
    const { attrs } = target;
    return attrs.id;
  }

  @OnChannel(...(process.env.TARGET_GUILDS?.split(',') || []))
  @UseCommand('rank [rank:string]', 'Ask for a rank')
  async rankCommand(
    @PutSession() session: Session,
    @PutArg(0) rank: string,
    @PutOption('targetId', '-t <targetId>') rawTargetId: string,
  ) {
    const targetId = this.parseTarget(rawTargetId);
    const rankUserId = targetId || session.userId;
    const actionDescription = `rank ${rank} to ${rankUserId} by ${session.userId}`;
    if (targetId) {
      const self = await this.getMember(session, session.userId);
      if (!self || self.roles[0] === 'member') {
        this.warn(
          `Skipped giving ${actionDescription} because of insufficient permissions.`,
        );
        return;
      }
      const target = await this.getMember(session, targetId);
      if (!target) {
        this.warn(
          `Skipped giving ${actionDescription} because of target not found.`,
        );
        return;
      }
    }
    rank ??= '';
    if (rank.length > 6) {
      this.warn(
        `Skipped giving ${actionDescription} because too long: ${rank.length}`,
      );
      return;
    }
    if (!targetId && !(await this.record.isCanGiveRecord(session))) {
      this.warn(
        `Skipped giving ${actionDescription} because already given within 2 hours.`,
      );
      return;
    }
    this.log(`Giving ${actionDescription}`);
    try {
      await (session.bot as unknown as OneBotBot).internal.setGroupSpecialTitle(
        session.guildId,
        rankUserId,
        rank,
      );
      await this.record.recordGiven(session, rank, targetId);
      this.log(`Given ${actionDescription}`);
    } catch (e) {
      this.error(`Error when giving ${actionDescription}: ${e.message}`);
    }
  }
}
