import { pickCert } from './check-cert';
import { Parser } from './parser';
import { getSiteNames } from './utility';
import md5 from 'nano-md5';
import fs from 'fs';
import { addSignCert, runSignCert } from './acme';

export interface SiteHttps {
  ports: string[];
  cert: string;
  redirect?: boolean;
  hsts?: boolean;
}

export interface Header {
  name: string;
  value: string;
}

export interface SiteRenderData {
  domains: string[];
  ports: string[];
  https?: SiteHttps;
  headers?: Header[];
  normalizeDomain?: string;
  disableTop?: boolean;
  serverExtra?: string;
  locationExtra?: string;
  locationExtraPre?: string;
  htpasswd?: string;
  cors?: boolean;
  acme?: boolean;
}

export interface ProxyRenderData extends SiteRenderData {
  proxy: true;
  minio?: boolean;
  timeout?: number;
  upstream: string;
  noVerifyCerts?: boolean;
  noBuffer?: boolean;
  sni?: boolean;
  sniName?: string;
  hostHeader?: string;
  noCache?: boolean;
}

interface FileRenderData extends SiteRenderData {
  root: string;
  index: string;
  expires: string;
}

export interface StaticRenderData extends FileRenderData {
  static: true;
  browse?: boolean;
}

export interface PhpRenderData extends FileRenderData {
  php: true;
  upstream: string;
  phpExtra?: string;
  phpExtraPre?: string;
}

export interface RedirectRenderData extends SiteRenderData {
  redirect: true;
  upstream: string;
  code: number;
}

export type SpecificRenderData =
  | PhpRenderData
  | StaticRenderData
  | ProxyRenderData
  | RedirectRenderData;

export interface RenderData {
  purgeAllowed?: string[];
  externalRealIp?: boolean;
  realIp?: string[];
  limitRate?: string;
  limitBurst?: string;
  maxCacheSize: string;
  dhparamPath: string;
  ticketKeyPath: string;
  certsPath: string;
  sites: SiteRenderData[];
  upstreams: Upstream[];
  httpExtra?: string;
  nginxExtra?: string;
  httpExtraPre?: string;
  nginxExtraPre?: string;
  acme?: boolean,
}

export interface Upstream {
  name: string;
  servers: string[];
  extra?: string;
}

const upstreams: Upstream[] = [];
function createUpstream(domain: string, urlInputs: string[], extra: string) {
  const urls = urlInputs.map((url) => new URL(url));
  if (urls.length === 1) {
    return urlInputs[0];
  }
  const name = `upstream_${domain.replace(/[^a-zA-Z0-9]/g, '_')}`;
  upstreams.push({
    name,
    servers: urls.map(
      (url) => `${url.host + (url.hash ? url.hash.replace(/[#&]/g, ' ') : '')}`,
    ),
    extra,
  });
  return `${urls[0].protocol}//${name}${urlInputs[0]
    .slice(urls[0].origin.length)
    .replace(urls[0].hash, '')}`;
}

async function getSiteData(
  domain: string,
  input: Record<string, string> = process.env,
): Promise<SpecificRenderData> {
  const parser = new Parser(`SITE_${domain}_`, input);
  const [hostname, port] = domain.split(':');
  const ports = [port || '80'];
  let https: SiteHttps;
  const httpsCert = parser.getString('HTTPS');
  const domains = hostname.split('+');
  if (httpsCert !== '0' && httpsCert !== 'false') {
    const cert = httpsCert?.startsWith('acme://')
      ? await addSignCert(domains, httpsCert)
      : !httpsCert ||
        httpsCert === '1' ||
        httpsCert === 'true' ||
        httpsCert === 'auto'
      ? await pickCert(domains)
      : httpsCert;
    if (cert) {
      if (port) {
        ports.pop();
      }
      https = {
        cert,
        ports: [port || '443'],
        redirect: !parser.getBoolean('HTTPS_NOREDIR') && !port,
        hsts: parser.getBoolean('HSTS'),
      };
    }
  }

  const targetUrlInput = input[`SITE_${domain}`];
  const targetUrlInputs = targetUrlInput.split(','); // TODO: support multiple targets as upstream
  const targetUrls = targetUrlInputs.map((url) => new URL(url));
  const targetUrl = targetUrls[0];
  let specificRenderData: SpecificRenderData;
  if (targetUrl.protocol === 'static:') {
    specificRenderData = {
      static: true,
      root: targetUrl.pathname,
      index: parser.getString('INDEX') || 'index.html',
      expires: parser.getString('EXPIRES') || '10m',
      browse: parser.getBoolean('BROWSE'),
    } as StaticRenderData;
  } else if (targetUrl.protocol === 'php:') {
    specificRenderData = {
      php: true,
      root: targetUrl.pathname,
      index: parser.getString('INDEX') || 'index.php',
      expires: parser.getString('EXPIRES') || '10m',
      upstream: targetUrl.host,
      phpExtra: parser.getString('PHP_EXTRA'),
      phpExtraPre: parser.getString('PHP_EXTRA_PRE'),
    } as PhpRenderData;
  } else if (targetUrl.protocol.startsWith('redirect+')) {
    specificRenderData = {
      redirect: true,
      upstream: targetUrl.href.slice(9),
      code: parser.getNumber('REDIRECT_CODE') || 301,
    } as RedirectRenderData;
  } else {
    const sni = parser.getString('SNI');
    specificRenderData = {
      proxy: true,
      upstream: createUpstream(
        domain,
        targetUrlInputs,
        parser.getString('UPSTREAM_EXTRA'),
      ),
      noVerifyCerts: parser.getBoolean('NO_VERIFY_CERTS'),
      noBuffer: parser.getBoolean('NO_BUFFER'),
      sni: sni === '1',
      sniName: sni === '1' ? undefined : sni,
      hostHeader: parser.getString('HOST'),
      noCache: parser.getBoolean('NO_CACHE'),
      minio: parser.getBoolean('MINIO'),
      timeout: parser.getNumber('TIMEOUT'),
    } as ProxyRenderData;
  }

  const basicPasswords = Object.entries(parser.getDict('HTPASSWD'));
  if (basicPasswords.length) {
    const htpasswd = basicPasswords
      .map(([user, pass]) => `${user}:${md5.crypt(pass)}`)
      .join('\n');
    await fs.promises.writeFile(
      `/etc/nginx/generated/htpasswd-${domain}`,
      htpasswd,
    );
  }

  return {
    domains: domains,
    ports,
    https,
    headers: Object.entries(parser.getDict('HEADER')).map(([name, value]) => ({
      name: name.replace(/_/g, '-'),
      value,
    })),
    normalizeDomain: parser.getString('NORMALIZE_DOMAIN'),
    disableTop: parser.getBoolean('DISABLE_TOP'),
    serverExtra: parser.getString('SERVER_EXTRA'),
    locationExtra: parser.getString('LOCATION_EXTRA'),
    locationExtraPre: parser.getString('LOCATION_EXTRA_PRE'),
    htpasswd: basicPasswords.length
      ? `/etc/nginx/generated/htpasswd-${domain}`
      : undefined,
    cors: parser.getBoolean('CORS'),
    acme: httpsCert?.startsWith('acme://'),
    ...specificRenderData,
  };
}

export async function getData(
  input: Record<string, string> = process.env,
  signCertPort = 80,
): Promise<RenderData> {
  const parser = new Parser('', input);
  const sites = await Promise.all(
    getSiteNames().map((domain) => getSiteData(domain, input)),
  );
  await runSignCert(signCertPort);
  return {
    purgeAllowed: parser.getArray('PURGE_ALLOWED'),
    externalRealIp: parser.getBoolean('EXTERNAL_REAL_IP'),
    realIp: parser.getArray('REAL_IP'),
    limitRate: parser.getString('LIMIT_RATE'),
    limitBurst: parser.getString('LIMIT_BURST'),
    maxCacheSize: parser.getString('MAX_CACHE_SIZE') || '10g',
    dhparamPath:
      parser.getString('DHPARAM_PATH') || '/etc/nginx/generated/dhparam.pem',
    ticketKeyPath:
      parser.getString('TICKET_KEY_PATH') || '/etc/nginx/generated/ticket.key',
    certsPath: parser.getString('CERTS_PATH') || '/etc/nginx/certs',
    sites,
    upstreams,
    httpExtra: parser.getString('HTTP_EXTRA'),
    nginxExtra: parser.getString('NGINX_EXTRA'),
    httpExtraPre: parser.getString('HTTP_EXTRA_PRE'),
    nginxExtraPre: parser.getString('NGINX_EXTRA_PRE'),
    acme: sites.some(s => s.acme),
  };
}
