import { X509Certificate } from 'crypto';
import * as fs from 'fs';
import path from 'path';
import _ from 'lodash';

fs.mkdirSync('/etc/nginx/certs', { recursive: true });
class Cert {
  public cert = new X509Certificate(
    fs.readFileSync(path.join('/etc/nginx/certs', this.dir, 'fullchain.pem')),
  );
  constructor(public dir: string) {}

  isNotExpired(acme = false) {
    const now = new Date();
    const validFrom = new Date(this.cert.validFrom);
    const validTo = new Date(this.cert.validTo);

    if (process.argv[2] === 'renewCert' && acme) {
      const threshold = new Date(validTo.getTime() - 7 * 24 * 60 * 60 * 1000);
      return now > validFrom && now < threshold;
    }

    return now > validFrom && now < validTo;
  }

  isOkWithDomain(domain: string) {
    return this.cert.checkHost(domain);
  }

  isOkWithDomains(domains: string[], acme = false) {
    return domains.every((domain) => this.isOkWithDomain(domain, acme));
  }
}

const certs = fs
  .readdirSync('/etc/nginx/certs')
  .filter((dir) =>
    fs.existsSync(path.join('/etc/nginx/certs', dir, 'fullchain.pem')),
  )
  .map((dir) => new Cert(dir))
  .filter((cert) => cert.isNotExpired());

export async function pickCert(domains: string[], acme = false) {
  const okCerts = certs.filter((cert) => cert.isOkWithDomains(domains, acme));
  if (!okCerts.length) {
    return;
  }
  return _.maxBy(okCerts, (cert) => cert.cert.validTo).dir;
}
