import { createServer } from 'http';
import acme, { Client } from 'acme-client';
import fs from 'fs';
import path from 'path';
import { pickCert } from './check-cert';

let email: string;
export const domainsToBeSigned: string[] = [];

export async function addSignCert(domains: string[], payload: string) {
  const pickedCert = await pickCert(domains, true);
  if (pickedCert) {
    return pickedCert;
  }
  domainsToBeSigned.push(...domains);
  if (!email) {
    // acme://
    email = payload.slice(7);
  }
  return domainsToBeSigned[0];
}

export async function runSignCert(signCertPort = 80) {
  if (!domainsToBeSigned.length) {
    return;
  }
  console.error(`Signing certificate for ${domainsToBeSigned.join(', ')}`);
  // sign a cert with acme-client

  const [certificateKey, certificateRequest] = await acme.crypto.createCsr({
    commonName: domainsToBeSigned[0],
    altNames: domainsToBeSigned.slice(1),
  });

  // store as token
  const contentMap = new Map<string, string>();
  const server = createServer(async (req, res) => {
    // read token from http://<YOUR_DOMAIN>/.well-known/acme-challenge/<TOKEN>
    const token = req.url?.split('/.well-known/acme-challenge/').pop();
    if (!token) {
      // 404
      res.writeHead(404);
      res.end('Token not Found');
      console.error(
        `${req.socket.remoteAddress}: Invalid request for ${req.url}`,
      );
      return;
    }
    const content = contentMap.get(token);
    if (!content) {
      // 404
      res.writeHead(404);
      res.end('Content not Found');
      console.error(
        `${req.socket.remoteAddress}: Token not found for ${token}`,
      );
      return;
    }
    res.writeHead(200);
    console.error(
      `${req.socket.remoteAddress}: Serving token for ${token}: ${content}`,
    );
    res.end(content);
  }).listen(signCertPort);
  const certDir = `/etc/nginx/certs/${domainsToBeSigned[0]}`;
  const accountFile = path.join(certDir, 'account.pem');
  const fullchainFile = path.join(certDir, 'fullchain.pem');
  const privkeyFile = path.join(certDir, 'privkey.pem');
  await fs.promises.mkdir(certDir, {
    recursive: true,
  });
  let accountKey: Buffer;
  try {
    accountKey = await fs.promises.readFile(accountFile);
  } catch (e) {
    accountKey = await acme.forge.createPrivateKey();
    await fs.promises.writeFile(accountFile, accountKey);
  }
  const acmeClient = new Client({
    directoryUrl: acme.directory.letsencrypt.production,
    accountKey,
  });

  try {
    const certificate = await acmeClient.auto({
      csr: certificateRequest,
      email,
      termsOfServiceAgreed: true,
      challengePriority: ['http-01'],
      challengeCreateFn: async (authz, challenge, keyAuthorization) => {
        if (challenge.type !== 'http-01') {
          return;
        }
        // store token
        console.error(
          `Storing token for ${challenge.token}: ${keyAuthorization}`,
        );
        contentMap.set(challenge.token, keyAuthorization);
      },
      challengeRemoveFn: async (authz, challenge, keyAuthorization) => {
        // does nothing
      },
    });
    console.error('Certificate signed');
    // save certificate as fullchain.pem and key as privkey.pem
    await fs.promises.writeFile(fullchainFile, certificate);
    await fs.promises.writeFile(privkeyFile, certificateKey);
  } catch (e) {
    console.error(
      `Failed to sign certificate for ${domainsToBeSigned.join(', ')}: ${
        e.stack
      }`,
    );
  }
  server.close();
}
