import * as fs from 'fs';
import Mustache from 'mustache';
import path from 'path';
import { getData } from './src/site';
import { execFile as _execFile } from 'child_process';
import { promisify } from 'util';

const execFile = promisify(_execFile);

const DEBUG = process.env.DEBUG === '1';
const NGINX_CONF_PATH = process.env.NGINX_CONF_PATH || '/etc/nginx/nginx.conf';

async function reloadNginx() {
  console.error('[nginx] Reloading...');
  try {
    const { stdout, stderr } = await execFile('nginx', ['-s', 'reload']);
    if (stdout?.trim()) console.error('[nginx][stdout]', stdout.trim());
    if (stderr?.trim()) console.error('[nginx][stderr]', stderr.trim());
    console.error('[nginx] Reloaded.');
  } catch (err: any) {
    console.error('[nginx] Reload failed:', err?.stderr || err?.message || err);
    throw err;
  }
}

function renderTemplate(parsed: any): string {
  const tpl = fs.readFileSync(
    path.join(__dirname, '..', 'views', 'nginx.conf.mustache'),
    'utf8',
  );
  const rendered = Mustache.render(tpl, parsed, undefined, {
    escape: (v) => v,
  });
  return rendered;
}

function writeConfig(rendered: string) {
  if (DEBUG) {
    // DEBUG=1：和原来一样，直接往 stdout 打
    console.error('[render] DEBUG=1, writing config to stdout instead of file');
    console.log(rendered);
  } else {
    fs.writeFileSync(NGINX_CONF_PATH, rendered, 'utf8');
    console.error('[render] Nginx config written to', NGINX_CONF_PATH);
  }
}

async function main() {
  try {
    const mode = process.argv[2];

    if (mode === 'renewCert') {
      console.error('[acme] Signing start');
      // 带超时的 getData
      const parsed = await getData(process.env as any, 61000);
      console.error('[acme] Signing done');

      console.error('[render] Nginx config render start (renewCert)');
      const rendered = renderTemplate(parsed);
      writeConfig(rendered);
      console.error('[render] Nginx config render done (renewCert)');

      if (!DEBUG) {
        await reloadNginx();
      } else {
        console.error('[nginx] DEBUG=1, skip reload');
      }
    } else {
      console.error('[render] Nginx config render start');
      const parsed = await getData(process.env as any);
      console.error('[render] parsed');
      const rendered = renderTemplate(parsed);
      writeConfig(rendered);
      console.error('[render] Nginx config render done');
    }

    process.exit(0);
  } catch (e) {
    console.error('[main] Error:', e);
    process.exit(1);
  }
}

main();
