// ========== 类型工具 ==========
type AwaitedLike<T> = T extends Promise<infer R> ? R : T;
type AnyFn = (...args: any[]) => any;

type Chain<T> = Promise<AwaitedLike<T>> &
  (T extends AnyFn
    ? (...a: Parameters<T>) => Chain<AwaitedLike<ReturnType<T>>>
    : {}) & {
    [K in keyof T]: T[K] extends (...a: infer A) => infer R
      ? (...a: A) => Chain<AwaitedLike<R>>
      : Chain<AwaitedLike<T[K]>>;
  };

// ========== 核心：不可变节点 ==========
class Node {
  private memo?: Promise<any>;
  constructor(
    private readonly source: any, // 只有根节点有意义；子节点可复用根的 source
    private readonly parent: Node | null, // 前缀
    private readonly op: ((x: any) => any) | null, // 从 parent 结果到本节点结果的变换
  ) {}

  run(): Promise<any> {
    if (!this.memo) {
      this.memo = (async () => {
        if (!this.parent) {
          // 根：把 source 解一次即可
          return await this.source;
        }
        // 先拿到父节点结果（会记忆化）
        const base = await this.parent.run();
        // 应用本节点的变换（允许返回同步或异步）
        return this.op ? this.op(base) : base;
      })();
    }
    return this.memo;
  }

  // 工厂：衍生子节点（分叉 = 复用同一个 parent）
  child(op: (x: any) => any): Node {
    return new Node(this.source, this, op);
  }

  // 拿到“相同节点”的句柄（显式 fork）
  fork(): Node {
    return new Node(this.source, this.parent, this.op);
  }
}

export const WF_NODE = Symbol('@@workflow/node');

function isWorkflowChain(x: any): x is Chain<any> {
  return !!x && typeof x === 'function' && x[WF_NODE] instanceof Node;
}

// ========== 对外 API：workflow ==========
export function workflow<T>(source: T | Promise<T>): Chain<T> {
  if (isWorkflowChain(source)) {
    return source as unknown as Chain<T>;
  }
  const root = new Node(source, null, null);
  return makeProxy<T>(root) as any;
}

// ========== 代理层：把 Node 暴露成链式 API ==========
function makeProxy<T>(node: Node): Chain<T> {
  const runOnce = () => node.run(); // 记忆化在 Node 内部

  const rootHandler: ProxyHandler<any> = {
    get(_t, prop) {
      if (prop === WF_NODE) return node;
      // 结束信号：所有 then/catch/finally 复用同一个 Promise
      if (prop === 'then')
        return (res: any, rej?: any) => runOnce().then(res, rej);
      if (prop === 'catch') return (rej: any) => runOnce().catch(rej);
      if (prop === 'finally') return (fin: any) => runOnce().finally(fin);

      // 访问一个属性：生成一个“Get”子节点，但不执行
      return linkProxy(prop);
    },
    // 支持对“当前值”直接二次调用： foo()()
    apply(_t, _this, argArray) {
      const child = node.child((cur: any) => {
        if (typeof cur !== 'function')
          throw new TypeError('Current value is not callable');
        return cur(...argArray);
      });
      return makeProxy<any>(child);
    },
  };

  const linkProxy = (prop: PropertyKey) =>
    new Proxy(function () {}, {
      // 把 “.foo(…args)” 记录为 Call
      apply(_t, _this, argArray) {
        const child = node.child((cur: any) => {
          const fn = (cur as any)[prop];
          if (typeof fn !== 'function') {
            throw new TypeError(`${String(prop)} is not a function`);
          }
          return fn.apply(cur, argArray);
        });
        return makeProxy<any>(child);
      },
      // 把 “.bar” 记录为 Get；继续深入时在这个 Get 的结果上再处理
      get(_t, next) {
        if (next === WF_NODE) return node;
        if (next === 'then')
          return (r: any, j?: any) =>
            node
              .child((cur: any) => (cur as any)[prop])
              .run()
              .then(r, j);
        if (next === 'catch')
          return (rej: any) =>
            node
              .child((cur: any) => (cur as any)[prop])
              .run()
              .catch(rej);
        if (next === 'finally')
          return (fin: any) =>
            node
              .child((cur: any) => (cur as any)[prop])
              .run()
              .finally(fin);

        const child = node.child((cur: any) => (cur as any)[prop]);
        return (makeProxy as any)(child)[next];
      },
    });

  return new Proxy(function () {}, rootHandler) as any;
}
