// dual-unified.ts
export type Dual<T> = T & PromiseLike<T>;

type ThenKey = 'then' | 'catch' | 'finally';
const isThenKey = (k: PropertyKey): k is ThenKey =>
  k === 'then' || k === 'catch' || k === 'finally';

type State = 'undecided' | 'pending' | 'fulfilled' | 'rejected';

/** 仅允许填入 “返回 Promise 的方法名” */
export type AsyncMethodKeys<T> = {
  [K in keyof T]-?: T[K] extends (...args: any[]) => Promise<any> ? K : never;
}[keyof T];

export interface DualizeOptions<T> {
  /** 这些方法在 undecided/pending 时会返回一个延迟执行函数，等待对象 Promise 完成后再调用 */
  asyncMethods?: readonly AsyncMethodKeys<T>[];
}

export function dualizeAny<T>(
  sync: () => T, // 同步构造；若抛错则视为 rejected
  asyncFn: () => Promise<T>, // 异步构造
  options?: DualizeOptions<T>,
): Dual<T> {
  let state: State = 'undecided';

  let value!: T; // fulfilled 时的值（含来自 sync 或 async）
  let reason: any; // rejected 的错误
  let p!: Promise<T>; // 缓存 Promise（resolved/rejected/进行中）

  const asyncMethodSet = new Set<PropertyKey>(
    (options?.asyncMethods ?? []) as readonly PropertyKey[],
  );

  const startAsync = () => {
    if (!p || state === 'undecided') {
      state = 'pending';
      p = Promise.resolve()
        .then(asyncFn)
        .then(
          (v) => {
            value = v;
            state = 'fulfilled';
            return v;
          },
          (e) => {
            reason = e;
            state = 'rejected';
            throw e;
          },
        );
    }
    return p;
  };

  const ensureSync = () => {
    if (state === 'undecided') {
      try {
        value = sync();
        state = 'fulfilled';
      } catch (e) {
        reason = e;
        state = 'rejected';
      }
    }
  };

  /** 在“对象可用”后调用某个异步方法（由 asyncMethods 声明） */
  const makeDeferredAsyncMethod =
    (prop: PropertyKey) =>
    (...args: any[]) =>
      startAsync().then((obj) => {
        const fn = (obj as any)[prop];
        return fn.apply(obj, args);
      });

  // 从某个值上取属性（原始值会装箱），并绑定 this
  const getFrom = (v: unknown, prop: PropertyKey) => {
    if (prop === Symbol.toPrimitive) {
      return (hint: 'default' | 'number' | 'string') => {
        const x: any = v;
        if (hint === 'number') return Number(x);
        if (hint === 'string') return String(x);
        if (typeof x === 'string') return x;
        const n = Number(x);
        return Number.isNaN(n) ? String(x) : n;
      };
    }
    if (prop === 'valueOf') return () => v as any;
    if (prop === 'toString') return () => String(v);

    const boxed: any =
      v !== null && (typeof v === 'object' || typeof v === 'function')
        ? v
        : Object(v as any);
    const out = boxed[prop];
    return typeof out === 'function' ? out.bind(boxed) : out;
  };

  const proxy = new Proxy(Object.create(null) as any, {
    get(_t, prop) {
      // then/catch/finally：走 Promise 通道
      if (isThenKey(prop)) {
        if (state === 'undecided') {
          startAsync();
        } else if (state === 'fulfilled') {
          // 若已 fulfilled（来自 sync 或 async），补一个已完成的 Promise
          p ||= Promise.resolve(value);
        } else if (state === 'rejected') {
          p ||= Promise.reject(reason);
        } else {
          // pending：已有 p
          startAsync();
        }
        const anyP: any = p;
        const m = anyP[prop];
        return typeof m === 'function' ? m.bind(anyP) : m;
      }

      // 声明为异步方法的键：在 undecided/pending 时返回“延迟函数”
      if (asyncMethodSet.has(prop)) {
        if (state === 'undecided' || state === 'pending') {
          startAsync();
          return makeDeferredAsyncMethod(prop);
        }
        if (state === 'fulfilled') {
          return getFrom(value, prop); // 同步可直接取到方法（其本身返回 Promise）
        }
        if (state === 'rejected') {
          // 访问即抛；也可以选择返回 () => Promise.reject(reason)
          throw reason;
        }
      }

      // 其它属性访问：遵循状态机
      switch (state) {
        case 'undecided': {
          ensureSync();
          // eslint-disable-next-line @typescript-eslint/ban-ts-comment
          // @ts-expect-error
          if (state === 'fulfilled') return getFrom(value, prop);
          // eslint-disable-next-line @typescript-eslint/ban-ts-comment
          // @ts-expect-error
          if (state === 'rejected') throw reason;
          // 理论上不会到这里
          throw new TypeError('Invalid state transition');
        }
        case 'pending': {
          // 非 asyncMethods 的属性在 pending 时不可同步读取
          throw new TypeError('Value is not ready yet. Please await it first.');
        }
        case 'fulfilled': {
          return getFrom(value, prop);
        }
        case 'rejected': {
          throw reason;
        }
      }
    },

    has(_t, key) {
      if (state === 'undecided') {
        ensureSync();
      }
      if (state === 'fulfilled') return key in Object(value as any);
      return false; // pending/rejected：保守处理
    },

    ownKeys() {
      if (state === 'undecided') ensureSync();
      if (state === 'fulfilled') return Reflect.ownKeys(Object(value as any));
      return [];
    },

    getOwnPropertyDescriptor(_t, key) {
      if (state === 'undecided') ensureSync();
      if (state === 'fulfilled')
        return Object.getOwnPropertyDescriptor(Object(value as any), key);
      return undefined;
    },
  });

  return proxy as Dual<T>;
}
