export class AbortedError extends Error {
  name = 'AbortError';
  constructor(
    msg = 'Operation aborted',
    public cause?: unknown,
  ) {
    super(msg);
  }
}

type AnyFunc = (...args: any[]) => any;
const isObjectLike = (v: unknown): v is object | AnyFunc =>
  (typeof v === 'object' && v !== null) || typeof v === 'function';

function boxPrimitive(v: unknown): object {
  return Object(v as any);
}

export interface AbortableOpts {
  /** 是否把原始值装箱后再代理（会改变 typeof/=== 语义，默认 false） */
  boxPrimitives?: boolean;
  /** 不递归代理子对象（仅代理顶层对象/函数，默认 false） */
  noRecursive?: boolean;
}

export function abortable<T>(
  obj: T,
  signal: AbortSignal,
  opts?: AbortableOpts,
): T {
  const { boxPrimitives = false, noRecursive = false } = opts ?? {};

  const throwIfAborted = () => {
    if (signal.aborted) {
      const r = (signal as any).reason;
      if (r instanceof Error) throw r;
      throw new AbortedError('Operation aborted', r);
    }
  };

  throwIfAborted();

  const targetToProxy = new WeakMap<object, any>();
  const proxyToTarget = new WeakMap<object, any>();

  const wrapNativePromise = <V>(pLike: PromiseLike<V>): Promise<V> => {
    const base = pLike instanceof Promise ? pLike : Promise.resolve(pLike);
    return base.then(
      async (v) => {
        throwIfAborted();
        return proxify(v);
      },
      async (e) => {
        throwIfAborted();
        throw e;
      },
    );
  };

  let shortCircuit = false;

  const proxify = <V>(value: V): V => {
    if (noRecursive) {
      if (shortCircuit) {
        return value;
      } else {
        shortCircuit = true;
      }
    }

    // Promise：按“真 Promise + 代理”的方案处理
    if (typeof (value as any).then === 'function') {
      return wrapNativePromise(value as any) as unknown as V;
    }

    // 原始值
    if (!isObjectLike(value)) {
      if (!boxPrimitives) return value;
      return proxify(boxPrimitive(value) as unknown as V);
    }

    // 复用
    const cached = targetToProxy.get(value as object);
    if (cached) return cached;

    const handler: ProxyHandler<any> = {
      get(target, p, receiver) {
        throwIfAborted();
        const out = Reflect.get(target, p, receiver);
        return proxify(out);
      },
      set(target, p, val, receiver) {
        throwIfAborted();
        return Reflect.set(target, p, val, receiver);
      },
      apply(target, thisArg, argArray) {
        throwIfAborted();
        const unwrappedThis =
          (isObjectLike(thisArg) && proxyToTarget.get(thisArg as any)) ||
          thisArg;
        const ret = Reflect.apply(target as AnyFunc, unwrappedThis, argArray);
        return proxify(ret);
      },
      construct(target, argArray, newTarget) {
        throwIfAborted();
        const instance = Reflect.construct(
          target as AnyFunc,
          argArray,
          newTarget,
        );
        return proxify(instance);
      },
      defineProperty(t, p, attrs) {
        throwIfAborted();
        return Reflect.defineProperty(t, p, attrs);
      },
      deleteProperty(t, p) {
        throwIfAborted();
        return Reflect.deleteProperty(t, p);
      },
      getOwnPropertyDescriptor(t, p) {
        throwIfAborted();
        return Reflect.getOwnPropertyDescriptor(t, p);
      },
      getPrototypeOf(t) {
        throwIfAborted();
        return Reflect.getPrototypeOf(t);
      },
      setPrototypeOf(t, proto) {
        throwIfAborted();
        return Reflect.setPrototypeOf(t, proto);
      },
      has(t, p) {
        throwIfAborted();
        return Reflect.has(t, p);
      },
      isExtensible(t) {
        throwIfAborted();
        return Reflect.isExtensible(t);
      },
      ownKeys(t) {
        throwIfAborted();
        return Reflect.ownKeys(t);
      },
      preventExtensions(t) {
        throwIfAborted();
        return Reflect.preventExtensions(t);
      },
    };

    const proxy = new Proxy(value as any, handler);
    targetToProxy.set(value as object, proxy);
    proxyToTarget.set(proxy, value);
    return proxy;
  };

  return proxify(obj);
}
