export interface DispatcherOptions {
  /** Max attempts per task. Default 3. */
  maxAttempts?: number;
  /** Backoff base in ms. backoff = base * 2^failCount. Default 1000. */
  backoffBaseMs?: number;
}

// Internal task record
type Task<F extends (...args: any[]) => Promise<any>> = {
  args: Parameters<F>;
  resolve: (v: Awaited<ReturnType<F>>) => void;
  reject: (e: any) => void;
  attempts: number;
  lastError?: any;
  triedWorkers: Set<number>; // for global dispatch
  specificWorkerIndex?: number; // for dispatchSpecific only
};

type WorkerState = {
  running: boolean;
  failCount: number;
  nextAvailableAt: number;
  totalRuns: number;
};

type RemovalWaiter = () => void;

type ActiveSlot<F extends (...args: any[]) => Promise<any>> = {
  kind: 'active';
  fn: F;
  state: WorkerState;
  queue: Task<F>[];
  removalWaiters: RemovalWaiter[];
  removed?: boolean; // marker when removed while running; retained for completion callback
};

type PendingSlot<F extends (...args: any[]) => Promise<any>> = {
  kind: 'pending';
  promise: Promise<F>;
  queue: Task<F>[];
  removalWaiters: RemovalWaiter[];
  error?: any;
  removed?: boolean; // if removed before resolved
};

type RejectedSlot<F extends (...args: any[]) => Promise<any>> = {
  kind: 'rejected';
  error: any;
  queue: Task<F>[];
  removalWaiters: RemovalWaiter[];
  removed?: boolean;
};

type Slot<F extends (...args: any[]) => Promise<any>> =
  | ActiveSlot<F>
  | PendingSlot<F>
  | RejectedSlot<F>;

export type WorkerSnapshot<F extends (...args: any[]) => Promise<any>> =
  | {
      index: number;
      status: 'active';
      fn: F;
      running: boolean;
      failCount: number;
      totalRuns: number;
      blockedMs: number;
      specificQueue: number;
    }
  | {
      index: number;
      status: 'pending';
      promise: Promise<F>;
      specificQueue: number;
    }
  | {
      index: number;
      status: 'rejected';
      error: string;
      specificQueue: number;
    };

export class WorkflowDispatcher<F extends (...args: any[]) => Promise<any>> {
  private readonly maxAttempts: number;
  private readonly backoffBaseMs: number;

  private readonly slots: Slot<F>[] = [];
  private readonly globalQueue: Task<F>[] = [];

  private pendingInits = 0;
  private everActivated = false;
  private drainScheduled = false;

  constructor(
    workersOrPromises: Array<F | Promise<F>>,
    options: DispatcherOptions = {},
  ) {
    // if (!workersOrPromises?.length) throw new Error('workers cannot be empty');

    this.maxAttempts = options.maxAttempts ?? 3;
    this.backoffBaseMs = options.backoffBaseMs ?? 1000;

    for (let i = 0; i < workersOrPromises.length; i++) {
      const w = workersOrPromises[i];
      if (typeof w === 'function') {
        const slot: ActiveSlot<F> = {
          kind: 'active',
          fn: w,
          state: {
            running: false,
            failCount: 0,
            nextAvailableAt: 0,
            totalRuns: 0,
          },
          queue: [],
          removalWaiters: [],
        };
        this.slots.push(slot);
        this.everActivated = true;
      } else {
        // Create a stable slot object and mutate it in-place on resolve/reject.
        this.pendingInits++;
        const slot: PendingSlot<F> = {
          kind: 'pending',
          promise: Promise.resolve(w),
          queue: [],
          removalWaiters: [],
        };
        this.slots.push(slot);

        slot.promise
          .then((fn) => {
            if (slot.removed) return; // was removed; ignore resolution
            // mutate in-place to active
            (slot as any).kind = 'active';
            (slot as any).fn = fn;
            (slot as any).state = {
              running: false,
              failCount: 0,
              nextAvailableAt: 0,
              totalRuns: 0,
            };
            this.everActivated = true;
            // keep queue & removalWaiters arrays as-is
          })
          .catch((err) => {
            if (slot.removed) return; // was removed; ignore
            // mutate in-place to rejected (keep queue/waiters)
            (slot as any).kind = 'rejected';
            (slot as any).error = err;
          })
          .finally(() => {
            this.pendingInits--;
            this.drain();
            if (this.pendingInits === 0 && !this.hasAnyActive()) {
              const err = new Error(
                'No workers available (all failed to initialize).',
              );
              setTimeout(() => this.rejectAllQueued(err), 0);
            }
          });
      }
    }

    if (this.everActivated) this.drain();
  }

  /** Dispatch: choose eligible active with least totalRuns; retry across workers on failure. */
  dispatch(...args: Parameters<F>): Promise<Awaited<ReturnType<F>>> {
    return new Promise((resolve, reject) => {
      const task: Task<F> = {
        args,
        resolve,
        reject,
        attempts: 0,
        triedWorkers: new Set(),
      };
      this.globalQueue.push(task);
      this.drain();
    });
  }

  /** Dispatch to a specific worker (ignore backoff), wait until it is free; retry on the same worker. */
  dispatchSpecific(
    index: number,
    ...args: Parameters<F>
  ): Promise<Awaited<ReturnType<F>>> {
    if (index < 0 || index >= this.slots.length) {
      return Promise.reject(new Error(`worker index out of range: ${index}`));
    }
    return new Promise((resolve, reject) => {
      const task: Task<F> = {
        args,
        resolve,
        reject,
        attempts: 0,
        triedWorkers: new Set(),
        specificWorkerIndex: index,
      };
      const slot = this.slots[index];
      slot.queue.push(task);
      this.drain();
    });
  }

  /** Replace a worker at index with a new active worker function. */
  public replaceWorker(index: number, fn: F): void {
    if (index < 0 || index >= this.slots.length) {
      throw new Error(`worker index out of range: ${index}`);
    }
    const prevHadActive = this.hasAnyActive();
    const slot = this.slots[index];

    // Preserve queue & removal waiters; reset failure/backoff; keep totalRuns to avoid skew.
    const preservedQueue = slot.queue;
    const preservedWaiters = slot.removalWaiters ?? [];

    const next: ActiveSlot<F> = {
      kind: 'active',
      fn,
      state: {
        running: false,
        failCount: 0,
        nextAvailableAt: 0,
        totalRuns: (slot as any).state?.totalRuns ?? 0,
      },
      queue: preservedQueue,
      removalWaiters: preservedWaiters,
    };

    // Mutate in-place if possible (keeps references stable), else replace array entry.
    Object.assign(slot as any, next);
    (slot as any).kind = 'active';
    (slot as any).fn = fn;
    (slot as any).state.failCount = 0;
    (slot as any).state.nextAvailableAt = 0;

    this.everActivated = true;
    if (!prevHadActive && this.hasAnyActive()) {
      this.drain();
    } else {
      this.drain();
    }
  }

  /** Add a new active worker at the tail; return its index. */
  public addWorker(fn: F): number {
    const slot: ActiveSlot<F> = {
      kind: 'active',
      fn,
      state: { running: false, failCount: 0, nextAvailableAt: 0, totalRuns: 0 },
      queue: [],
      removalWaiters: [],
    };
    const index = this.slots.length;
    this.slots.push(slot);
    this.everActivated = true;
    this.drain();
    return index;
  }

  /**
   * Remove a worker completely (splice). It becomes unavailable immediately.
   * Returns a Promise that resolves when its last running task (if any) finishes.
   */
  public removeWorker(index: number): Promise<void> {
    if (index < 0 || index >= this.slots.length) {
      return Promise.reject(new Error(`worker index out of range: ${index}`));
    }

    const slot = this.slots[index];

    // Reject all queued specific tasks on this worker (macro-task to avoid unhandled)
    const queued = slot.queue.splice(0);
    const removalErr = new Error(`Worker[${index}] removed`);
    setTimeout(() => {
      for (const t of queued) t.reject(removalErr);
    }, 0);

    // Decide completion promise:
    let completion: Promise<void>;
    const isRunning = slot.kind === 'active' && slot.state.running;
    if (!isRunning) {
      completion = Promise.resolve();
    } else {
      completion = new Promise<void>((resolve) => {
        slot.removalWaiters.push(resolve);
      });
    }

    // Mark as removed (so any pending init resolution is ignored)
    (slot as any).removed = true;

    // Physically remove the slot
    this.slots.splice(index, 1);

    // Re-map indices in all remaining tasks:
    // 1) Fix specificWorkerIndex in every remaining slot.queue
    for (let i = 0; i < this.slots.length; i++) {
      const s = this.slots[i];
      for (const t of s.queue) {
        if (typeof t.specificWorkerIndex === 'number') {
          if (t.specificWorkerIndex === index) {
            // This should not happen because we just removed and flushed its queue,
            // but guard anyway.
            t.reject(new Error(`Worker[${index}] no longer exists`));
          } else if (t.specificWorkerIndex > index) {
            t.specificWorkerIndex -= 1;
          }
        }
      }
    }
    // 2) Fix triedWorkers sets in globalQueue
    for (const t of this.globalQueue) {
      if (t.triedWorkers.has(index)) t.triedWorkers.delete(index);
      const next = new Set<number>();
      for (const w of t.triedWorkers) {
        next.add(w > index ? w - 1 : w);
      }
      t.triedWorkers = next;
    }

    // Trigger scheduling for the remaining system
    this.drain();

    return completion;
  }

  snapshot(): WorkerSnapshot<F>[] {
    const now = Date.now();

    return this.slots.map((slot, i) => {
      switch (slot.kind) {
        case 'active': {
          const s = slot.state;
          return {
            index: i,
            status: 'active' as const,
            fn: slot.fn,
            running: s.running,
            failCount: s.failCount,
            totalRuns: s.totalRuns,
            blockedMs: Math.max(0, s.nextAvailableAt - now),
            specificQueue: slot.queue.length,
          };
        }

        case 'pending':
          return {
            index: i,
            status: 'pending' as const,
            promise: slot.promise,
            specificQueue: slot.queue.length,
          };

        case 'rejected':
          return {
            index: i,
            status: 'rejected' as const,
            error: String(slot.error ?? 'unknown error'),
            specificQueue: slot.queue.length,
          };
      }
    });
  }

  get pending(): number {
    return this.globalQueue.length;
  }

  // ---------------- scheduling ----------------

  private drain() {
    if (this.drainScheduled) return;
    this.drainScheduled = true;
    queueMicrotask(() => {
      this.drainScheduled = false;
      this._drainLoop();
    });
  }

  private _drainLoop() {
    // If no active workers and still initializing, wait; if all inited and none active, constructor already rejects all.
    if (!this.hasAnyActive()) {
      if (this.pendingInits > 0) return;
      return;
    }

    // First: flush rejected workers' specific queues (macro-task rejection)
    for (let i = 0; i < this.slots.length; i++) {
      const slot = this.slots[i];
      if (slot.kind === 'rejected' && slot.queue.length > 0) {
        const q = slot.queue.splice(0);
        const err = new Error(
          `Worker[${i}] failed to initialize: ${String(slot.error ?? 'unknown error')}`,
        );
        setTimeout(() => {
          for (const t of q) t.reject(err);
        }, 0);
      }
    }

    let progressed = true;
    while (progressed) {
      progressed = false;

      // 1) Run specific queues for active workers (ignore backoff)
      for (let i = 0; i < this.slots.length; i++) {
        const slot = this.slots[i];
        if (slot.kind !== 'active') continue;
        const st = slot.state;
        if (!st.running && slot.queue.length > 0) {
          const task = slot.queue.shift()!;
          this.startTaskOnActiveSlot(i, slot, task, /*fromSpecific*/ true);
          progressed = true;
        }
      }

      // 2) Run global queue (choose eligible active with least totalRuns)
      if (this.globalQueue.length > 0) {
        const idx = this.pickBestActiveForGlobal();
        if (idx !== -1) {
          const slot = this.slots[idx] as ActiveSlot<F>;
          const task = this.globalQueue.shift()!;
          this.startTaskOnActiveSlot(idx, slot, task, /*fromSpecific*/ false);
          progressed = true;
        }
      }
    }
  }

  private hasAnyActive() {
    return this.slots.some((s) => s.kind === 'active');
  }

  private rejectAllQueued(err: any) {
    while (this.globalQueue.length) this.globalQueue.shift()!.reject(err);
    for (const slot of this.slots) {
      while (slot.queue.length) slot.queue.shift()!.reject(err);
    }
  }

  private isEligibleActive(i: number, now: number) {
    const slot = this.slots[i] as ActiveSlot<F>;
    const slotGood = (s: ActiveSlot<F>) =>
      s.kind === 'active' && !s.state?.running;
    if (!slotGood(slot)) return false;
    if (now >= slot.state.nextAvailableAt) return true;
    return !this.slots.some((other: ActiveSlot<F>, j) => {
      if (j === i) return false;
      if (!slotGood(other)) return false;
      return now >= other.state.nextAvailableAt;
    });
  }

  private pickBestActiveForGlobal(): number {
    const now = Date.now();
    const task = this.globalQueue[0];

    // Prefer actives not tried yet
    let best = -1,
      bestRuns = Infinity;
    for (let i = 0; i < this.slots.length; i++) {
      if (!this.isEligibleActive(i, now)) continue;
      if (task?.triedWorkers.has(i)) continue;
      const slot = this.slots[i] as ActiveSlot<F>;
      if (slot.state.totalRuns < bestRuns) {
        bestRuns = slot.state.totalRuns;
        best = i;
      }
    }
    if (best !== -1) return best;

    // Allow already-tried actives
    best = -1;
    bestRuns = Infinity;
    for (let i = 0; i < this.slots.length; i++) {
      if (!this.isEligibleActive(i, now)) continue;
      const slot = this.slots[i] as ActiveSlot<F>;
      if (slot.state.totalRuns < bestRuns) {
        bestRuns = slot.state.totalRuns;
        best = i;
      }
    }
    return best;
  }

  private startTaskOnActiveSlot(
    index: number,
    slot: ActiveSlot<F>,
    task: Task<F>,
    fromSpecific: boolean,
  ) {
    const st = slot.state;
    st.running = true;
    st.totalRuns += 1;

    const finalize = () => {
      st.running = false;
      // If someone is waiting for this worker to finish (removeWorker), resolve them when idle.
      if (
        slot.removalWaiters.length > 0 &&
        !st.running &&
        slot.queue.length === 0
      ) {
        const list = slot.removalWaiters.splice(0);
        for (const w of list) w();
      }
      this.drain();
    };

    (async () => {
      try {
        const result = await slot.fn(...task.args);
        st.failCount = Math.max(0, st.failCount - 1);
        task.resolve(result as Awaited<ReturnType<F>>);
      } catch (err) {
        task.lastError = err;
        st.failCount += 1;
        st.nextAvailableAt =
          Date.now() + this.backoffBaseMs * Math.pow(2, st.failCount);

        if (fromSpecific) {
          task.attempts += 1;
          if (task.attempts >= this.maxAttempts) {
            task.reject(task.lastError);
          } else {
            // retry on the same worker (ignore backoff)
            slot.queue.push(task);
          }
        } else {
          task.attempts += 1;
          task.triedWorkers.add(index);
          const activeCount = this.slots.filter(
            (s) => s.kind === 'active',
          ).length;
          const allActiveTriedOnce = task.triedWorkers.size >= activeCount;
          const attemptsLimitReached = task.attempts >= this.maxAttempts;

          if (allActiveTriedOnce || attemptsLimitReached) {
            task.reject(task.lastError);
          } else {
            this.globalQueue.push(task);
          }
        }
      } finally {
        finalize();
      }
    })();
  }
}
