// __tests__/workflow-dispatcher-extend.spec.ts
import { WorkflowDispatcher } from '../src/workflow-dispatcher';

type F = (x: number) => Promise<string>;

function makeSuccess(label: string): F {
  const fn = jest.fn(async (x: number) => `${label}:${x}`);
  return fn as F;
}
function makeAlwaysFail(label: string): F {
  const fn = jest.fn(async () => {
    throw new Error(`fail:${label}`);
  });
  return fn as F;
}
function deferred<T>() {
  let resolve!: (v: T) => void;
  let reject!: (e: any) => void;
  const promise = new Promise<T>((res, rej) => {
    resolve = res;
    reject = rej;
  });
  return { promise, resolve, reject };
}
async function flushMicro(n = 2) {
  for (let i = 0; i < n; i++) await Promise.resolve();
}
async function nextMacrotask() {
  await new Promise((r) => setTimeout(r, 0));
}

describe('replaceWorker()', () => {
  test('replaces a pending worker to active and drains immediately if it becomes the first active', async () => {
    const Adef = deferred<F>(); // slot[0] pending initially
    const d = new WorkflowDispatcher<F>([Adef.promise], { backoffBaseMs: 10 });

    // queue a specific task to slot 0 while pending
    const p = d.dispatchSpecific(0, 1);
    await flushMicro();

    // replace pending with active fn; should start running and resolve
    d.replaceWorker(0, makeSuccess('A'));
    await flushMicro();
    await expect(p).resolves.toBe('A:1');

    // then a global dispatch should also use A
    await expect(d.dispatch(2)).resolves.toBe('A:2');
  });

  test('replaces an active worker with a new fn, resets backoff but keeps totalRuns', async () => {
    const bad = makeAlwaysFail('Old');
    const good = makeSuccess('New');
    const d = new WorkflowDispatcher<F>([bad], { backoffBaseMs: 10 });

    // first dispatch will fail and be thrown (only one worker, hits maxAttempts=3 eventually)
    await expect(d.dispatch(1)).rejects.toThrow(/fail:Old/);

    const before = (d.snapshot()[0] as any).totalRuns;
    d.replaceWorker(0, good);

    // should succeed with new fn
    await expect(d.dispatch(2)).resolves.toBe('New:2');

    const after = (d.snapshot()[0] as any).totalRuns;
    expect(after).toBeGreaterThanOrEqual(before + 1); // totalRuns not reset to 0
  });
});

describe('addWorker()', () => {
  test('adds a new active worker and immediately helps drain queued tasks', async () => {
    // slot[0] will be a long-running worker to block
    const gate = deferred<void>();
    const longRunner: F = jest.fn(async (x: number) => {
      await gate.promise;
      return `L:${x}`;
    });
    const d = new WorkflowDispatcher<F>([longRunner], { backoffBaseMs: 10 });

    // occupy slot[0]
    const p1 = d.dispatch(1);
    await flushMicro();

    // second task will queue (no free worker)
    const p2 = d.dispatch(2);
    await flushMicro();

    // add a new fast worker at tail
    const idx = d.addWorker(makeSuccess('N'));
    expect(idx).toBe(1);

    // p2 should finish via the new worker immediately
    await expect(p2).resolves.toBe('N:2');

    // release p1 and it should finish too
    gate.resolve();
    await expect(p1).resolves.toBe('L:1');
  });
});

describe('removeWorker()', () => {
  test('removing a pending worker splices it and re-maps indices (specific queues + global triedWorkers)', async () => {
    // slots: [pending A, active B, active C]
    const Adef = deferred<F>();
    const B = makeSuccess('B');
    const C = makeSuccess('C');
    const d = new WorkflowDispatcher<F>([Adef.promise, B, C], {
      backoffBaseMs: 10,
    });

    // queue specific to index 0 (pending)
    const pA1 = d.dispatchSpecific(0, 100);
    await flushMicro();
    // remove index 0 (pending A)
    const removed = d.removeWorker(0);
    await nextMacrotask(); // allow macro rejection for its queue
    await expect(removed).resolves.toBeUndefined();
    await expect(pA1).rejects.toThrow(/removed|failed to initialize/);

    // now original [1,2] -> become [0,1]
    // specific to "original 1" should now be index 0 and succeed
    await expect(d.dispatchSpecific(0, 1)).resolves.toBe('B:1');
    await expect(d.dispatchSpecific(1, 2)).resolves.toBe('C:2');

    // check globalQueue triedWorkers re-map:
    // trigger a fail on B to add it to triedWorkers of a global task
    const badB: F = jest.fn(async () => {
      throw new Error('fail:B');
    });
    d.replaceWorker(0, badB);
    const g = d.dispatch(7); // will try slot[0] then retry others
    await flushMicro();
    // now remove the failing worker (index 0)
    const done = d.removeWorker(0);
    await nextMacrotask();
    await expect(done).resolves.toBeUndefined();
    // global task should still complete using C (now at index 0 after splice)
    await expect(g).resolves.toBe('C:7');
  });

  test('removing an active running worker resolves when that last task finishes', async () => {
    // slot[0] long running, slot[1] fast
    const gate = deferred<void>();
    const long: F = jest.fn(async (x: number) => {
      await gate.promise;
      return `L:${x}`;
    });
    const fast = makeSuccess('F');
    const d = new WorkflowDispatcher<F>([long, fast], { backoffBaseMs: 10 });

    // occupy slot[0] with a specific task so we know exactly which worker
    const p1 = d.dispatchSpecific(0, 1);
    await flushMicro();

    // remove slot[0] while it is running -> removal promise should resolve only after p1 settles
    const removing = d.removeWorker(0);

    // the slot is no longer pickable; new specific(0) should now refer to old index 1 (fast)
    await expect(d.dispatchSpecific(0, 2)).resolves.toBe('F:2');

    // still running p1 should finish, then `removing` resolves
    const settleOrder: string[] = [];
    p1.then(() => settleOrder.push('p1'));
    removing.then(() => settleOrder.push('removing'));

    // release long running
    gate.resolve();
    await flushMicro();
    await nextMacrotask();

    expect(settleOrder).toEqual(['p1', 'removing']);
  });

  test('removing an idle active worker resolves immediately and re-maps indices correctly', async () => {
    const A = makeSuccess('A');
    const B = makeSuccess('B');
    const C = makeSuccess('C');
    const d = new WorkflowDispatcher<F>([A, B, C], { backoffBaseMs: 10 });

    // Nothing running yet, remove middle index 1 (B)
    const pr = d.removeWorker(1);
    await expect(pr).resolves.toBeUndefined();

    // Now original C becomes index 1
    await expect(d.dispatchSpecific(0, 10)).resolves.toBe('A:10');
    await expect(d.dispatchSpecific(1, 20)).resolves.toBe('C:20');

    // Global dispatch still balances across remaining two
    const r1 = await d.dispatch(1);
    const r2 = await d.dispatch(2);
    expect([r1, r2].some((s) => s.startsWith('A'))).toBe(true);
    expect([r1, r2].some((s) => s.startsWith('C'))).toBe(true);
  });

  test('removing a rejected worker resolves immediately and flushes its specific queue', async () => {
    // Build: [rejected X, active Y]
    const Xdef = deferred<F>();
    const Y = makeSuccess('Y');
    const d = new WorkflowDispatcher<F>([Xdef.promise, Y], {
      backoffBaseMs: 10,
    });

    // specific to 0 queues into X (pending)
    const p = d.dispatchSpecific(0, 1);
    await flushMicro();

    // reject X init
    Xdef.reject(new Error('X-init-fail'));
    await nextMacrotask(); // allow rejected pending flush in drain()

    // remove the rejected worker
    const pr = d.removeWorker(0);
    await expect(pr).resolves.toBeUndefined();

    // the queued task should have already been rejected
    await expect(p).rejects.toThrow(/failed to initialize|removed/);

    // now only Y remains as index 0
    await expect(d.dispatchSpecific(0, 2)).resolves.toBe('Y:2');
  });
});
