// i18n.spec.ts

import {
  createI18nLookupMiddleware,
  I18n,
  I18nDictionary,
  I18nLookupMiddleware,
} from '../src/i18n';

type Ctx = {
  dict?: I18nDictionary;
  name?: string;
};

describe('I18n - end-to-end behavior', () => {
  const locales = ['en', 'zh', 'zh-Hans', 'zh-Hans-CN'] as const;

  const baseDict: I18nDictionary = {
    en: { hello: 'Hello', price: '', bye: 'Bye' },
    'zh-Hans': { hello: '你好', user: '用户：${name}' },
    zh: { hi: '嗨' },
  };

  const makeI18n = () =>
    new I18n<[ctx?: Ctx]>({
      locales: locales as unknown as string[],
      defaultLocale: 'en',
    });

  it('translates placeholders via lookup middleware with hierarchy fallback', async () => {
    const i18n = makeI18n();
    i18n.middleware(
      I18nLookupMiddleware<[Ctx]>(() => baseDict, { matchType: 'hierarchy' }),
    );

    const s = await i18n.translateString(
      'zh-Hans-CN',
      'Say: #{hello} / #{price} / #{miss}',
    );
    expect(s).toBe('Say: 你好 /  / #{miss}');
  });

  it('keeps raw string unchanged when no placeholders exist', async () => {
    const i18n = makeI18n();
    i18n.middleware(
      I18nLookupMiddleware(() => baseDict, { matchType: 'hierarchy' }),
    );

    const s = await i18n.translateString('en', 'Plain text only');
    expect(s).toBe('Plain text only');
  });

  it('prefers exact locale over hierarchy fallback', async () => {
    const i18n = makeI18n();
    const dict: I18nDictionary = {
      zh: { hello: '嗨(zh)' },
      'zh-Hans': { hello: '你好(zh-Hans)' },
    };
    i18n.middleware(
      I18nLookupMiddleware(() => dict, { matchType: 'hierarchy' }),
    );

    const s1 = await i18n.translateString('zh-Hans', '=> #{hello}');
    expect(s1).toBe('=> 你好(zh-Hans)');

    const s2 = await i18n.translateString('zh-Hans-CN', '=> #{hello}');
    expect(s2).toBe('=> 你好(zh-Hans)'); // fell back from zh-Hans-CN to zh-Hans
  });

  it('supports startsWith strategy and picks the longest matching prefix', async () => {
    const i18n = makeI18n();
    const dict: I18nDictionary = {
      zh: { greet: '嗨(zh)' },
      'zh-Hans': { greet: '嗨(zh-Hans)' },
    };
    i18n.middleware(
      I18nLookupMiddleware(() => dict, { matchType: 'startsWith' }),
    );

    const s = await i18n.translateString('zh-Hans-CN', '=> #{greet}');
    expect(s).toBe('=> 嗨(zh-Hans)'); // chooses the longest prefix "zh-Hans"
  });

  it('awaits async dictionary factory', async () => {
    const i18n = makeI18n();
    const dict: I18nDictionary = { en: { hello: 'Hello(async)' } };

    i18n.middleware(
      I18nLookupMiddleware(async () => {
        await new Promise((r) => setTimeout(r, 10));
        return dict;
      }),
    );

    const s = await i18n.translateString('en', 'Result: #{hello}');
    expect(s).toBe('Result: Hello(async)');
  });

  it('respects empty string translations', async () => {
    const i18n = makeI18n();
    const dict: I18nDictionary = { en: { price: '' } };
    i18n.middleware(I18nLookupMiddleware(() => dict));

    const s = await i18n.translateString('en', 'Price: #{price}');
    expect(s).toBe('Price: '); // empty string is a valid hit
  });

  it('leaves placeholder intact for missing keys', async () => {
    const i18n = makeI18n();
    i18n.middleware(I18nLookupMiddleware(() => baseDict));

    const s = await i18n.translateString('en', 'Unknown: #{nope}');
    expect(s).toBe('Unknown: #{nope}');
  });

  it('middleware order: prior insertion runs first and can short-circuit', async () => {
    const i18n = makeI18n();

    const hits: string[] = [];

    // normal dictionary lookup at tail
    i18n.middleware(
      I18nLookupMiddleware(() => baseDict, { matchType: 'hierarchy' }),
    );

    // prior middleware: short-circuit for a specific key
    i18n.middleware((locale, key, next) => {
      hits.push(`prior:${key}`);
      if (key === 'special') return 'SPECIAL!';
      return next();
    }, true);

    const a = await i18n.translateString(
      'zh-Hans-CN',
      'A #{special} B #{hello}',
    );
    expect(a).toBe('A SPECIAL! B 你好');

    expect(hits).toEqual(['prior:special', 'prior:hello']); // even when not short-circuiting, prior runs before others
  });

  it('removeMiddleware detaches the correct middleware', async () => {
    const i18n = makeI18n();
    const logs: string[] = [];

    const mwA = (locale: string, key: string, next: any) => {
      logs.push('A');
      return next();
    };
    const mwB = (locale: string, key: string, next: any) => {
      logs.push('B');
      return next();
    };
    const mwC = I18nLookupMiddleware(() => baseDict, {
      matchType: 'hierarchy',
    });

    i18n.middleware(mwA);
    i18n.middleware(mwB);
    i18n.middleware(mwC);

    i18n.removeMiddleware(mwB); // remove middle

    const s = await i18n.translateString('zh-Hans-CN', '=> #{hello}');
    expect(s).toBe('=> 你好');
    expect(logs).toEqual(['A']); // B was removed, so never logged
  });

  it('translate() preserves prototype and translates own data properties', async () => {
    class Person {
      constructor(
        public first: string,
        public last: string,
      ) {}
      get full() {
        return `${this.first} ${this.last}`;
      }
    }
    const p = new Person('Foo #{hello}', 'Bar');

    const i18n = makeI18n();
    i18n.middleware(
      I18nLookupMiddleware(() => baseDict, { matchType: 'hierarchy' }),
    );

    const out = await i18n.translate('zh-Hans-CN', p);
    expect(Object.getPrototypeOf(out)).toBe(Person.prototype);
    expect(out.first).toBe('Foo 你好');
    expect(out.last).toBe('Bar');
    // accessors are copied as-is; translateObjectPreservingProto mutates via setter only if present
    expect(out.full).toBe('Foo 你好 Bar');
  });

  it('translate() keeps built-in objects as-is and handles cycles', async () => {
    const i18n = makeI18n();
    i18n.middleware(
      I18nLookupMiddleware(() => baseDict, { matchType: 'hierarchy' }),
    );

    const cyc: any = { a: 'X #{hello}', date: new Date(), rx: /abc/ };
    cyc.self = cyc; // cycle

    const out = await i18n.translate('zh-Hans-CN', cyc);
    expect(out.a).toBe('X 你好');
    expect(out.date).toBe(cyc.date);
    expect(out.rx).toBe(cyc.rx);
    expect(out.self).toBe(cyc); // already visited -> original reference (not deeply cloned)
  });

  it('translate() resolves Promises and translates resolved values', async () => {
    const i18n = makeI18n();
    i18n.middleware(
      I18nLookupMiddleware(() => baseDict, { matchType: 'hierarchy' }),
    );

    const obj = {
      title: Promise.resolve('Head #{hello}'),
      nested: {
        x: Promise.resolve('N #{bye}'),
      },
    };

    const out = await i18n.translate('zh-Hans-CN', obj);
    expect(await out.title).toBe('Head 你好'); // promise value gets translated after resolution
    expect(await out.nested.x).toBe('N Bye');
  });

  it('createI18nLookupMiddleware returns a generic-bound factory', async () => {
    const i18n = makeI18n();

    // bind Ex = [Ctx] once
    const makeLookup = createI18nLookupMiddleware<[Ctx]>();
    const lookup = makeLookup(async (locale, key, ctx) => ctx.dict!, {
      matchType: 'hierarchy',
    });

    i18n.middleware(lookup);

    const s = await i18n.translateString('zh-Hans-CN', 'User: #{user}', {
      dict: baseDict,
      name: 'Karin',
    });

    // note: our dict has '用户：${name}', but we didn't add a formatter middleware here,
    // so ${name} stays literal. The point of this test is that generic binding works and lookup hits.
    expect(s).toBe('User: 用户：${name}');
  });

  it('placeholders not found remain intact while raw segments are preserved', async () => {
    const i18n = makeI18n();
    i18n.middleware(
      I18nLookupMiddleware(() => baseDict, { matchType: 'hierarchy' }),
    );

    const s = await i18n.translateString(
      'en',
      'Alpha #{missing} Beta #{hello} Gamma',
    );
    expect(s).toBe('Alpha #{missing} Beta Hello Gamma');
  });
});
