import { Configurer } from '../src/configurer/configurer';

type TestConfig = {
  HOST: string;
  PORT: string;
  ENABLE_RECONNECT: string;
  NO_CONNECT_COUNT_LIMIT: string;
  ALT_VERSIONS: string;
  FLOAT_VALUES: string;
  BOOL_VALUES: string;
  JSON_CONFIG: string;
};

const defaultConfig: TestConfig = {
  HOST: '::',
  PORT: '7911',
  ENABLE_RECONNECT: '1',
  NO_CONNECT_COUNT_LIMIT: '',
  ALT_VERSIONS: '2330,2331',
  FLOAT_VALUES: '1.5,2.75',
  BOOL_VALUES: '1,0,true,false,null',
  JSON_CONFIG: '{"mode":"default","retry":3}',
};

describe('Configurer', () => {
  test('loadConfig merges with priority: default < obj < env', () => {
    const configurer = new Configurer(defaultConfig);
    const instance = configurer.loadConfig({
      obj: {
        host: '0.0.0.0',
        PORT: 9000,
      },
      env: {
        PORT: '10000',
      },
    });

    expect(instance.getString('HOST')).toBe('0.0.0.0');
    expect(instance.getString('PORT')).toBe('10000');
  });

  test('getBoolean uses boolean default parsed from default config with defaultValue=false', () => {
    const configurer = new Configurer(defaultConfig);

    const withEmptyReconnect = configurer.loadConfig({
      env: {
        ENABLE_RECONNECT: '',
      },
    });
    expect(withEmptyReconnect.getBoolean('ENABLE_RECONNECT')).toBe(true);

    const withEmptyCountLimit = configurer.loadConfig({
      env: {
        NO_CONNECT_COUNT_LIMIT: '',
      },
    });
    expect(withEmptyCountLimit.getBoolean('NO_CONNECT_COUNT_LIMIT')).toBe(
      false,
    );
  });

  test('getInt/getFloat/get arrays parse from getString', () => {
    const configurer = new Configurer(defaultConfig);
    const instance = configurer.loadConfig({
      env: {
        PORT: '8080',
        ALT_VERSIONS: '3000,abc,3002',
        FLOAT_VALUES: '3.5,NaN,4.25',
        BOOL_VALUES: '1,0,true,false,,null',
      },
    });

    expect(instance.getInt('PORT')).toBe(8080);
    expect(instance.getFloat('PORT')).toBe(8080);
    expect(instance.getStringArray('ALT_VERSIONS')).toEqual([
      '3000',
      'abc',
      '3002',
    ]);
    expect(instance.getIntArray('ALT_VERSIONS')).toEqual([3000, 3002]);
    expect(instance.getFloatArray('FLOAT_VALUES')).toEqual([3.5, 4.25]);
    expect(instance.getBooleanArray('BOOL_VALUES')).toEqual([
      true,
      false,
      true,
      false,
      true,
      false,
    ]);
  });

  test('getString keeps whitespace string value', () => {
    const configurer = new Configurer(defaultConfig);
    const instance = configurer.loadConfig({
      env: {
        HOST: '   ',
      },
    });

    expect(instance.getString('HOST')).toBe('   ');
  });

  test('getJSON parses JSON and falls back to default for empty value', () => {
    const configurer = new Configurer(defaultConfig);
    const withCustom = configurer.loadConfig({
      env: {
        JSON_CONFIG: '{"mode":"custom","retry":1}',
      },
    });
    const withEmpty = configurer.loadConfig({
      env: {
        JSON_CONFIG: '  ',
      },
    });

    expect(withCustom.getJSON<{ mode: string; retry: number }>('JSON_CONFIG')).toEqual({
      mode: 'custom',
      retry: 1,
    });
    expect(withEmpty.getJSON<{ mode: string; retry: number }>('JSON_CONFIG')).toEqual({
      mode: 'default',
      retry: 3,
    });
  });

  test('getJSON throws when JSON parse fails', () => {
    const configurer = new Configurer(defaultConfig);
    const instance = configurer.loadConfig({
      env: {
        JSON_CONFIG: '{invalid json}',
      },
    });

    expect(() => instance.getJSON('JSON_CONFIG')).toThrow(
      'Failed to parse JSON config "JSON_CONFIG"',
    );
  });

  test('normalizeConfigValue treats object as JSON string', () => {
    const configurer = new Configurer(defaultConfig);
    const instance = configurer.loadConfig({
      obj: {
        jsonConfig: { mode: 'obj', retry: 8 },
      },
    });

    expect(instance.getString('JSON_CONFIG')).toBe('{"mode":"obj","retry":8}');
  });

  test('generateExampleObject converts key to camelCase and parses number/array/json-object values', () => {
    const configurer = new Configurer(defaultConfig);
    const example = configurer.generateExampleObject();

    expect(example.host).toBe('::');
    expect(example.port).toBe(7911);
    expect(example.altVersions).toEqual([2330, 2331]);
    expect(example.floatValues).toEqual(['1.5', '2.75']);
    expect(example.jsonConfig).toEqual({ mode: 'default', retry: 3 });
  });
});
