export class ConfigurerInstance<T extends Record<string, string>> {
  constructor(
    public config: T,
    private readonly defaultConfig: T,
  ) {}

  getString<K extends keyof T>(key: K): string {
    return (this.config[key] || this.defaultConfig[key]) as string;
  }

  getInt<K extends keyof T>(key: K): number {
    return parseInt(this.getString(key));
  }

  getFloat<K extends keyof T>(key: K): number {
    return parseFloat(this.getString(key));
  }

  getBoolean<K extends keyof T>(key: K): boolean {
    const defaultBoolean = parseConfigBoolean(this.defaultConfig[key], false);
    return parseConfigBoolean(this.getString(key), defaultBoolean);
  }

  getStringArray<K extends keyof T>(key: K): string[] {
    return convertStringArray(this.getString(key));
  }

  getIntArray<K extends keyof T>(key: K): number[] {
    return convertIntArray(this.getString(key));
  }

  getFloatArray<K extends keyof T>(key: K): number[] {
    return convertFloatArray(this.getString(key));
  }

  getBooleanArray<K extends keyof T>(key: K): boolean[] {
    const defaultBoolean = parseConfigBoolean(this.defaultConfig[key], false);
    return convertBooleanArray(this.getString(key), defaultBoolean);
  }

  getJSON<R = unknown, K extends keyof T = keyof T>(key: K): R {
    const value = this.getString(key);
    const jsonString =
      typeof value === 'string' && !value.trim()
        ? (this.defaultConfig[key] as string)
        : value;
    try {
      return JSON.parse(jsonString) as R;
    } catch (error) {
      const message =
        error instanceof Error ? `: ${error.message}` : '';
      throw new Error(
        `Failed to parse JSON config "${String(key)}"${message}`,
      );
    }
  }
}

export class Configurer<T extends Record<string, string>> {
  constructor(public defaultConfig: T) {}

  loadConfig(
    options: { env?: Record<string, string | undefined>; obj?: any } = {},
  ): ConfigurerInstance<T> {
    const readConfig =
      options?.obj && typeof options.obj === 'object' ? options.obj : {};
    const normalizedConfig = normalizeConfigByDefaultKeys(
      readConfig,
      this.defaultConfig,
    );

    return new ConfigurerInstance<T>(
      {
        ...this.defaultConfig,
        ...normalizedConfig,
        ...(options?.env || {}),
      } as T,
      this.defaultConfig,
    );
  }

  generateExampleObject(): Record<string, unknown> {
    return Object.fromEntries(
      Object.entries(this.defaultConfig).map(([key, value]) => {
        const typedValue = toTypedValue(value);
        if (
          typedValue &&
          typeof typedValue === 'object' &&
          !Array.isArray(typedValue)
        ) {
          return [toCamelCaseKey(key), typedValue];
        }
        if (value.includes(',')) {
          return [
            toCamelCaseKey(key),
            value.split(',').map((v) => toTypedValue(v)),
          ];
        }
        return [toCamelCaseKey(key), typedValue];
      }),
    );
  }
}

export type TypeFromConfigurer<C extends ConfigurerInstance<any>> =
  C extends ConfigurerInstance<infer T> ? T : never;

function toCamelCaseKey(key: string): string {
  const lower = key.toLowerCase();
  return lower.replace(/_([a-z0-9])/g, (_, ch: string) => ch.toUpperCase());
}

function normalizeConfigValue(value: unknown): string | undefined {
  if (value == null) {
    return undefined;
  }
  if (typeof value === 'string') {
    return value;
  }
  if (typeof value === 'number') {
    return value.toString();
  }
  if (typeof value === 'boolean') {
    return value ? '1' : '0';
  }
  if (Array.isArray(value)) {
    return value.map((item) => normalizeArrayItem(item)).join(',');
  }
  if (typeof value === 'object') {
    return JSON.stringify(value);
  }
  return String(value);
}

function normalizeArrayItem(value: unknown): string {
  if (typeof value === 'string') {
    return value;
  }
  if (typeof value === 'number') {
    return value.toString();
  }
  if (typeof value === 'boolean') {
    return value ? '1' : '0';
  }
  return String(value);
}

function normalizeConfigByDefaultKeys<T extends Record<string, string>>(
  readConfig: Record<string, unknown>,
  defaultConfig: T,
): Partial<T> {
  const normalizedConfig: Partial<T> = {};
  for (const key of Object.keys(defaultConfig) as Array<keyof T>) {
    const rawKey = key as string;
    const camelKey = toCamelCaseKey(rawKey);
    const value =
      readConfig[camelKey] !== undefined
        ? readConfig[camelKey]
        : readConfig[rawKey];
    const normalized = normalizeConfigValue(value);
    if (normalized !== undefined) {
      normalizedConfig[key] = normalized as T[typeof key];
    }
  }
  return normalizedConfig;
}

function parseConfigBoolean(value: unknown, defaultValue = false): boolean {
  if (typeof value === 'boolean') {
    return value;
  }
  if (typeof value === 'number') {
    return value !== 0;
  }
  if (typeof value === 'string') {
    const normalized = value.trim().toLowerCase();
    if (defaultValue) {
      return !(
        normalized === '0' ||
        normalized === 'false' ||
        normalized === 'null'
      );
    }
    return !(
      normalized === '' ||
      normalized === '0' ||
      normalized === 'false' ||
      normalized === 'null'
    );
  }
  if (value == null) {
    return defaultValue;
  }
  return Boolean(value);
}

function convertStringArray(str: string): string[] {
  return (
    str
      ?.split(',')
      .map((s) => s.trim())
      .filter((s) => s) || []
  );
}

function convertIntArray(str: string): number[] {
  return (
    str
      ?.split(',')
      .map((s) => parseInt(s.trim()))
      .filter((n) => !isNaN(n)) || []
  );
}

function convertFloatArray(str: string): number[] {
  return (
    str
      ?.split(',')
      .map((s) => parseFloat(s.trim()))
      .filter((n) => !isNaN(n)) || []
  );
}

function convertBooleanArray(str: string, defaultValue = false): boolean[] {
  return (
    str
      ?.split(',')
      .map((s) => parseConfigBoolean(s.trim(), defaultValue))
      .filter((item) => typeof item === 'boolean') || []
  );
}

function toTypedValue(value: string): string | number | Record<string, unknown> {
  const trimmed = value.trim();
  if (/^\d+$/.test(trimmed)) {
    return Number.parseInt(trimmed, 10);
  }
  const jsonObject = parseJsonObjectString(trimmed);
  if (jsonObject !== undefined) {
    return jsonObject;
  }
  return trimmed;
}

function parseJsonObjectString(
  value: string,
): Record<string, unknown> | undefined {
  if (!value.startsWith('{') || !value.endsWith('}')) {
    return undefined;
  }
  try {
    const parsed = JSON.parse(value);
    if (parsed && typeof parsed === 'object' && !Array.isArray(parsed)) {
      return parsed as Record<string, unknown>;
    }
  } catch {
    return undefined;
  }
  return undefined;
}
