import { AnyClass, Empty } from '../types';
import {
  AppContext,
  AppServiceClass,
  AppProvideArgs,
  AppProvidedMerged,
  AppProvideOptions,
  AppContextUsed,
  AppContextState,
} from './types';

const ProvidePrefix = 'provide:';

const getMethodDescriptor = (cls: AnyClass, key: PropertyKey) => {
  let proto = cls.prototype;
  while (proto && proto !== Object.prototype) {
    const desc = Object.getOwnPropertyDescriptor(proto, key as any);
    if (desc) return desc;
    proto = Object.getPrototypeOf(proto);
  }
  return undefined;
};

type ProvideRecord = {
  classRef: AnyClass;
  factory: (ctx: AppContextCore<any, any>) => any;
};

type LoadEntry = {
  classRef: AnyClass;
  inst: any;
};

type ObjectStep = (ctx: AppContextCore<any, any>) => void;

export class AppContextCore<Cur = Empty, Req = Empty> {
  private __current: Cur;
  private __required: Req;

  private provideRecords: ProvideRecord[] = [];
  private registry = new Map<string | AnyClass, LoadEntry>();
  private objectSteps: ObjectStep[] = [];
  private parentContexts = new Set<AppContextCore<any, any>>();
  started = false;
  private starting = false;
  private startingEntries: LoadEntry[] | null = null;
  private createdRecords: Set<ProvideRecord> | null = null;
  private createdEntries: Map<ProvideRecord, LoadEntry> | null = null;
  private loadingRecords = new Set<ProvideRecord>();

  private findProvideRecord(key: AnyClass): ProvideRecord | undefined {
    for (let i = 0; i < this.provideRecords.length; i += 1) {
      const record = this.provideRecords[i];
      if (
        record.classRef === key &&
        (!this.createdRecords || !this.createdRecords.has(record))
      ) {
        return record;
      }
    }
    return undefined;
  }

  private createEntryFromRecord(record: ProvideRecord): LoadEntry {
    const existing = this.createdEntries?.get(record);
    if (existing) {
      return existing;
    }

    if (this.loadingRecords.has(record)) {
      throw new Error(
        `Circular dependency detected while providing: ${record.classRef?.name ?? 'UnknownService'}`,
      );
    }

    this.loadingRecords.add(record);
    try {
      const entry: LoadEntry = {
        classRef: record.classRef,
        inst: record.factory(this),
      };
      this.createdRecords?.add(record);
      this.createdEntries?.set(record, entry);
      this.registry.set(record.classRef, entry);
      this.startingEntries?.push(entry);
      return entry;
    } finally {
      this.loadingRecords.delete(record);
    }
  }

  private applyUsedContext(other: AppContextCore<any, any>) {
    // Copy provide records
    if (Array.isArray(other?.provideRecords)) {
      this.provideRecords.push(...other.provideRecords);
    }

    // Copy and apply object steps
    if (Array.isArray(other?.objectSteps)) {
      this.objectSteps.push(...other.objectSteps);
      for (const step of other.objectSteps) {
        step(this);
      }
    }

    // If the other context has already started, copy loaded entries only.
    // They should remain initialized by the source context and must not be re-initialized.
    if (other?.started) {
      if (other?.registry instanceof Map) {
        for (const [key, value] of other.registry.entries()) {
          this.registry.set(key, value);
        }
      }
    }
  }

  private hasStartedInParentChain(
    visited = new Set<AppContextCore<any, any>>(),
  ): boolean {
    if (visited.has(this)) return false;
    visited.add(this);
    if (this.started) return true;
    for (const parent of this.parentContexts) {
      if (parent.hasStartedInParentChain(visited)) {
        return true;
      }
    }
    return false;
  }

  private propagateUsedContextToParents(
    other: AppContextCore<any, any>,
    visited = new Set<AppContextCore<any, any>>(),
  ) {
    if (visited.has(this)) return;
    visited.add(this);
    for (const parent of this.parentContexts) {
      parent.applyUsedContext(other);
      parent.propagateUsedContextToParents(other, visited);
    }
  }

  provide<
    C extends AppServiceClass<Cur, Req>,
    const P extends string = '',
    const M extends (keyof InstanceType<C>)[] = [],
  >(
    cls: C,
    ...args: AppProvideArgs<Cur, Req, C, P, M>
  ): AppProvidedMerged<Cur, Req, C, P, M> {
    const last = args[args.length - 1] as any;
    const hasOptions =
      !!last &&
      typeof last === 'object' &&
      ('provide' in last ||
        'merge' in last ||
        'useValue' in last ||
        'useFactory' in last ||
        'useClass' in last);

    const options = (hasOptions ? last : undefined) as
      | AppProvideOptions<Cur, Req, C, P, M>
      | undefined;

    const _args = (
      hasOptions ? args.slice(0, -1) : args
    ) as ConstructorParameters<C>;

    const classRef = cls as unknown as AnyClass;

    // Create factory function that will be called during start() with the target ctx.
    const factory = (ctx: AppContextCore<any, any>) =>
      options?.useValue ??
      (options?.useFactory
        ? options.useFactory(ctx as any, ..._args)
        : new (options?.useClass ?? cls)(ctx as any, ..._args));

    // Record the provide configuration
    this.provideRecords.push({
      classRef,
      factory,
    });

    // Set up property accessors if needed
    if (options?.provide) {
      const prop = options.provide;
      const step: ObjectStep = (ctx) => {
        Object.defineProperty(ctx, prop, {
          enumerable: true,
          configurable: true,
          get: () => {
            const currentEntry = ctx.registry.get(classRef);
            return currentEntry?.inst;
          },
        });
      };
      step(this);
      this.objectSteps.push(step);
    }

    if (options?.merge?.length) {
      for (const key of options.merge) {
        const desc = getMethodDescriptor(cls, key);
        const isMethod = !!desc && typeof desc.value === 'function';

        const step: ObjectStep = (ctx) => {
          if (isMethod) {
            Object.defineProperty(ctx, key, {
              enumerable: true,
              configurable: true,
              get: () => {
                const currentEntry = ctx.registry.get(classRef);
                const currentInst = currentEntry?.inst;
                const fn = (currentInst as any)?.[key];
                return typeof fn === 'function' ? fn.bind(currentInst) : fn;
              },
            });
            return;
          }

          Object.defineProperty(ctx, key, {
            enumerable: true,
            configurable: true,
            get: () => {
              const currentEntry = ctx.registry.get(classRef);
              return (currentEntry?.inst as any)?.[key];
            },
            set: (value: any) => {
              const currentEntry = ctx.registry.get(classRef);
              if (!currentEntry) return;
              (currentEntry.inst as any)[key] = value;
            },
          });
        };
        step(this);
        this.objectSteps.push(step);
      }
    }

    return this as any;
  }

  get<R>(
    cls:
      | AppServiceClass<Cur, Req, any, R>
      | (() => AppServiceClass<Cur, Req, any, R>),
  ): R {
    let key = cls as unknown as AnyClass;
    if (
      !this.registry.has(key) &&
      typeof cls === 'function' &&
      !(cls as any).prototype
    ) {
      key = (cls as () => AppServiceClass<Cur, Req, any, R>)() as AnyClass;
    }

    if (!this.registry.has(key) && this.starting) {
      const record = this.findProvideRecord(key);
      if (record) {
        this.createEntryFromRecord(record);
      }
    }

    if (!this.registry.has(key)) {
      throw new Error(`Service not provided: ${key?.name ?? cls.name}`);
    }

    const entry = this.registry.get(key)!;
    return entry.inst as any;
  }

  /**
   * @deprecated Use get() instead. getAsync() is no longer needed as all services are loaded synchronously during start().
   */
  async getAsync<R>(
    cls:
      | AppServiceClass<Cur, Req, any, R>
      | (() => AppServiceClass<Cur, Req, any, R>),
  ): Promise<R> {
    return this.get(cls);
  }

  use<const Ctxes extends AppContext<any, any>[]>(
    ...ctxes: Ctxes
  ): AppContextUsed<Cur, Req, Ctxes> {
    for (const ctx of ctxes) {
      const other = ctx as any as AppContextCore<any, any>;

      if (this.hasStartedInParentChain() && !other?.started) {
        throw new Error(
          'Cannot use an unstarted context into a started context.',
        );
      }

      this.applyUsedContext(other);
      other.parentContexts.add(this);
      this.propagateUsedContextToParents(other);
    }

    return this as any;
  }

  define(): AppContext<Cur, Req> {
    return this as any;
  }

  async start(): Promise<Empty extends Req ? AppContext<Cur, Req> : never> {
    if (this.started) {
      return this as any;
    }

    const startedEntries: LoadEntry[] = [];
    const preloadedKeys = new Set(this.registry.keys());
    this.starting = true;
    this.startingEntries = startedEntries;
    this.createdRecords = new Set();
    this.createdEntries = new Map();
    try {
      // Create all instances. Dependencies requested via get() during construction
      // can be created on-demand from remaining provide records.
      for (const record of this.provideRecords) {
        if (preloadedKeys.has(record.classRef)) {
          continue;
        }
        this.createEntryFromRecord(record);
      }

      // Resolve all promises created in this start().
      for (const entry of startedEntries) {
        if (entry.inst && typeof entry.inst.then === 'function') {
          entry.inst = await entry.inst;
        }
      }

      // Init only instances created in this start().
      for (const entry of startedEntries) {
        const inst = entry.inst;
        if (inst && typeof inst.init === 'function') {
          await inst.init();
        }
      }
      this.started = true;
      return this as any;
    } finally {
      this.createdEntries = null;
      this.createdRecords = null;
      this.startingEntries = null;
      this.starting = false;
    }
  }
}

export const createAppContext = <Req = Empty>() =>
  new AppContextCore<Empty, Req>() as AppContext<Empty, Req>;
