import { Configurer } from '../src/configurer/configurer';

type TestConfig = {
  HOST: string;
  PORT: string;
  ENABLE_RECONNECT: string;
  NO_CONNECT_COUNT_LIMIT: string;
  ALT_VERSIONS: string;
  FLOAT_VALUES: string;
  BOOL_VALUES: string;
};

const defaultConfig: TestConfig = {
  HOST: '::',
  PORT: '7911',
  ENABLE_RECONNECT: '1',
  NO_CONNECT_COUNT_LIMIT: '',
  ALT_VERSIONS: '2330,2331',
  FLOAT_VALUES: '1.5,2.75',
  BOOL_VALUES: '1,0,true,false,null',
};

describe('Configurer', () => {
  test('loadConfig merges with priority: default < obj < env', () => {
    const configurer = new Configurer(defaultConfig);
    const instance = configurer.loadConfig({
      obj: {
        host: '0.0.0.0',
        PORT: 9000,
      },
      env: {
        PORT: '10000',
      },
    });

    expect(instance.getString('HOST')).toBe('0.0.0.0');
    expect(instance.getString('PORT')).toBe('10000');
  });

  test('getBoolean uses boolean default parsed from default config with defaultValue=false', () => {
    const configurer = new Configurer(defaultConfig);

    const withEmptyReconnect = configurer.loadConfig({
      env: {
        ENABLE_RECONNECT: '',
      },
    });
    expect(withEmptyReconnect.getBoolean('ENABLE_RECONNECT')).toBe(true);

    const withEmptyCountLimit = configurer.loadConfig({
      env: {
        NO_CONNECT_COUNT_LIMIT: '',
      },
    });
    expect(withEmptyCountLimit.getBoolean('NO_CONNECT_COUNT_LIMIT')).toBe(
      false,
    );
  });

  test('getInt/getFloat/get arrays parse from getString', () => {
    const configurer = new Configurer(defaultConfig);
    const instance = configurer.loadConfig({
      env: {
        PORT: '8080',
        ALT_VERSIONS: '3000,abc,3002',
        FLOAT_VALUES: '3.5,NaN,4.25',
        BOOL_VALUES: '1,0,true,false,,null',
      },
    });

    expect(instance.getInt('PORT')).toBe(8080);
    expect(instance.getFloat('PORT')).toBe(8080);
    expect(instance.getStringArray('ALT_VERSIONS')).toEqual([
      '3000',
      'abc',
      '3002',
    ]);
    expect(instance.getIntArray('ALT_VERSIONS')).toEqual([3000, 3002]);
    expect(instance.getFloatArray('FLOAT_VALUES')).toEqual([3.5, 4.25]);
    expect(instance.getBooleanArray('BOOL_VALUES')).toEqual([
      true,
      false,
      true,
      false,
      true,
      false,
    ]);
  });

  test('generateExampleObject converts key to camelCase and parses number/array values', () => {
    const configurer = new Configurer(defaultConfig);
    const example = configurer.generateExampleObject();

    expect(example.host).toBe('::');
    expect(example.port).toBe(7911);
    expect(example.altVersions).toEqual([2330, 2331]);
    expect(example.floatValues).toEqual(['1.5', '2.75']);
  });
});
