import { createAppContext, AppContext } from '../src/app-context';

const delay = (ms: number) => new Promise<void>((r) => setTimeout(r, ms));

type Equal<A, B> =
  (<T>() => T extends A ? 1 : 2) extends <T>() => T extends B ? 1 : 2
    ? true
    : false;
type Expect<T extends true> = T;

class CounterService {
  constructor(
    public ctx: AppContext,
    public value = 1,
  ) {}

  inc() {
    this.value += 1;
    return this.value;
  }

  async ping(add: number) {
    return this.value + add;
  }
}

class AsyncMutableService {
  constructor(public ctx: AppContext) {}
  count = 0;
}

class InitLogService {
  constructor(
    public ctx: AppContext,
    private name: string,
    private logs: string[],
  ) {}

  async init() {
    this.logs.push(`init:${this.name}`);
  }
}

class InitCounterService {
  constructor(
    public ctx: AppContext,
    private counter: { value: number },
  ) {}

  async init() {
    this.counter.value += 1;
  }
}

class NeedsMergedMethodService {
  value: number;

  constructor(public ctx: AppContext) {
    if (typeof (ctx as any).inc !== 'function') {
      throw new Error('missing merged method: inc');
    }
    this.value = (ctx as any).inc();
  }
}

class NeedsCounterService {
  counter: CounterService;

  constructor(public ctx: AppContext) {
    this.counter = ctx.get(CounterService);
  }
}

describe('app-context runtime', () => {
  test('provide + merge(method) binds this correctly', async () => {
    const ctx = await createAppContext()
      .provide(CounterService, 1, { merge: ['inc'] })
      .define()
      .start();

    const inc = ctx.inc;
    expect(inc()).toBe(2);
    expect(inc()).toBe(3);
  });

  test('async factory is resolved during start()', async () => {
    const ctx = await createAppContext()
      .provide(CounterService, 5, {
        provide: 'counter',
        useFactory: async (self, ...args: unknown[]) => {
          const initial = args[args.length - 1];
          await delay(20);
          return new CounterService(self, initial as number);
        },
      })
      .define()
      .start();

    // After start(), all async services are resolved
    expect(ctx.counter.value).toBe(5);
    await expect(ctx.counter.ping(3)).resolves.toBe(8);
  });

  test('merge(property) can be set after start()', async () => {
    const ctx = await createAppContext()
      .provide(AsyncMutableService, {
        merge: ['count'],
        useFactory: async (self) => {
          await delay(20);
          return new AsyncMutableService(self);
        },
      })
      .define()
      .start();

    // After start(), properties can be set directly
    ctx.count = 42;
    expect(ctx.get(AsyncMutableService).count).toBe(42);
    expect(ctx.count).toBe(42);
  });

  test('use replays object definition steps and merges registry', async () => {
    const ctx1 = createAppContext()
      .provide(CounterService, 7, { provide: 'counter', merge: ['inc'] })
      .define();
    const root = await createAppContext().use(ctx1).define().start();

    expect(root.counter.inc()).toBe(8);
    expect(root.get(CounterService)).toMatchObject({
      value: 8,
    });
  });

  test('start resolves async provides and runs init in registration order', async () => {
    const logs: string[] = [];
    const ctx1 = createAppContext()
      .provide(InitLogService, 'A', logs, {
        useFactory: async (self, ...args: unknown[]) => {
          const [name, output] = args.length >= 2 ? args : [args[0], logs];
          const resolvedName = String(name);
          const resolvedOutput = output as string[];
          await delay(20);
          resolvedOutput.push(`resolve:${resolvedName}`);
          return new InitLogService(self, resolvedName, resolvedOutput);
        },
      })
      .define();
    const ctx2 = createAppContext()
      .provide(InitLogService, 'B', logs, {
        useFactory: async (self, ...args: unknown[]) => {
          const [name, output] = args.length >= 2 ? args : [args[0], logs];
          const resolvedName = String(name);
          const resolvedOutput = output as string[];
          await delay(5);
          resolvedOutput.push(`resolve:${resolvedName}`);
          return new InitLogService(self, resolvedName, resolvedOutput);
        },
      })
      .define();

    await createAppContext().use(ctx1).use(ctx2).define().start();
    expect(logs).toContain('init:A');
    expect(logs).toContain('init:B');
    expect(logs.indexOf('init:A')).toBeLessThan(logs.indexOf('init:B'));
  });

  test('started context cannot use unstarted context', async () => {
    const started = await createAppContext().define().start();
    const unstarted = createAppContext()
      .provide(CounterService, 1, { provide: 'counter' })
      .define();

    expect(() => started.use(unstarted)).toThrow(
      'Cannot use an unstarted context into a started context.',
    );
  });

  test('using started context does not re-init imported providers', async () => {
    const counter = { value: 0 };
    const startedChild = await createAppContext()
      .provide(InitCounterService, counter)
      .define()
      .start();

    expect(counter.value).toBe(1);

    const root = await createAppContext().use(startedChild).define().start();
    expect(counter.value).toBe(1);
    expect(root.get(InitCounterService)).toBe(startedChild.get(InitCounterService));
  });

  test('provider from used context can access merged members on target context', async () => {
    const parent = createAppContext()
      .provide(CounterService, 10, { merge: ['inc'] })
      .define();
    const child = createAppContext()
      .provide(NeedsMergedMethodService, { provide: 'needsMerged' })
      .define();

    const root = await createAppContext().use(parent).use(child).define().start();
    expect(root.needsMerged.value).toBe(11);
  });

  test('provider in earlier used context can get provider from later used context', async () => {
    const a = createAppContext()
      .provide(NeedsCounterService, { provide: 'needsCounter' })
      .define();
    const b = createAppContext()
      .provide(CounterService, 21, { provide: 'counter' })
      .define();

    const root = await createAppContext().use(a).use(b).define().start();
    expect(root.needsCounter.counter).toBe(root.counter);
    expect(root.needsCounter.counter.value).toBe(21);
  });

  test('provider in later used context can get provider from earlier used context', async () => {
    const a = createAppContext()
      .provide(CounterService, 34, { provide: 'counter' })
      .define();
    const b = createAppContext()
      .provide(NeedsCounterService, { provide: 'needsCounter' })
      .define();

    const root = await createAppContext().use(a).use(b).define().start();
    expect(root.needsCounter.counter).toBe(root.counter);
    expect(root.needsCounter.counter.value).toBe(34);
  });
});

describe('app-context type checks', () => {
  test('compile-time type assertions', async () => {
    const ctx = await createAppContext()
      .provide(CounterService, 1, { provide: 'counter', merge: ['inc'] })
      .define()
      .start();

    const n: number = ctx.counter.inc();
    expect(n).toBe(2);

    type _counter = Expect<Equal<typeof ctx.counter, CounterService>>;
    type _inc = Expect<Equal<typeof ctx.inc, () => number>>;
    const ok: _counter | _inc = true;
    expect(ok).toBe(true);
  });

  test('start return type is never when requirement not fulfilled', () => {
    const reqCtx = createAppContext<{ foo: number }>().define();
    type Ret = Awaited<ReturnType<typeof reqCtx.start>>;
    type _ret = Expect<Equal<Ret, never>>;
    const ok: _ret = true;
    expect(ok).toBe(true);
  });

  test('use merges context types', async () => {
    const a = createAppContext()
      .provide(CounterService, 1, { provide: 'counter' })
      .define();
    const b = await createAppContext().use(a).define().start();
    const v: number = b.counter.value;
    expect(v).toBe(1);
  });
});
