import { ProtoMiddlewareDispatcher } from '../src/middleware-dispatcher';

describe('ProtoMiddlewareDispatcher', () => {
  class Base {
    value: number;
  }
  class Sub extends Base {
    get mutated() { 
      return this.value * 2;
    }
  }

  it('builds middlewares by prototype chain (prior: Base->Sub, normal: Sub->Base)', async () => {
    const d = new ProtoMiddlewareDispatcher<[res: number]>();
    const order: string[] = [];

    d.middleware(
      Base,
      async (inst, x, next) => {
        order.push('base:prior1');
        return next();
      },
      true,
    );
    d.middleware(
      Base,
      async (inst, x, next) => {
        order.push('base:prior2');
        return next();
      },
      true,
    );
    d.middleware(
      Sub,
      async (inst, x, next) => {
        order.push('sub:prior');
        return next();
      },
      true,
    );
    d.middleware(Sub, async (inst, x, next) => {
      order.push('sub:normal0');
      return next();
    });
    d.middleware(Sub, async (inst, x, next) => {
      order.push('sub:normal');
      expect(inst).toBeInstanceOf(Sub);
      expect(inst.value).toBe(5);
      const newSub = new Sub();
      newSub.value = inst.value + x;
      return next(newSub, 555); // override args
    });
    d.middleware(Base, async (inst, x, next) => {
      order.push('base:normal');
      expect(inst).toBeInstanceOf(Base);
      expect(inst.value).toBe(8); // 5 + 3
      expect(x).toBe(555);
      ++inst.value;
      return next();
    });

    const sub = new Sub();
    sub.value = 5;
    const res = await d.dispatch(sub, 3);
    expect(order).toEqual([
      'base:prior2',
      'base:prior1',
      'sub:prior',
      'sub:normal0',
      'sub:normal',
      'base:normal',
    ]);
    expect(res).toBeInstanceOf(Sub);
    expect(res.value).toBe(9); // 5 + 3 + 1
    expect(res.mutated).toBe(18);
  });
});
