// i18n-lookup.spec.ts

import { I18nLookupMiddleware } from '../src/i18n';
import { MiddlewareDispatcher } from '../src/middleware-dispatcher/middleware-dispatcher';

type F = (locale: string, key: string) => string | undefined;

describe('I18nLookupMiddleware - matching strategies', () => {
  it('exact strategy uses only exact branch', async () => {
    const dict = {
      en: { k: 'EN' },
      'zh-Hans': { k: 'ZH-Hans' },
    };
    const mw = I18nLookupMiddleware(() => dict, { matchType: 'exact' });

    const d = new MiddlewareDispatcher<F>({
      acceptResult: (res) => res != null,
    });
    d.middleware(mw);

    const a = await d.dispatch('zh-Hans-CN', 'k'); // no exact "zh-Hans-CN"
    expect(a).toBeUndefined();

    const b = await d.dispatch('zh-Hans', 'k');
    expect(b).toBe('ZH-Hans');
  });

  it('hierarchy strategy finds closest branch by reducing segments', async () => {
    const dict = {
      zh: { k: 'ZH' },
      'zh-Hans': { k: 'ZH-Hans' },
    };
    const d = new MiddlewareDispatcher<F>({
      acceptResult: (res) => res != null,
    });
    d.middleware(I18nLookupMiddleware(() => dict, { matchType: 'hierarchy' }));

    const a = await d.dispatch('zh-Hans-CN', 'k');
    expect(a).toBe('ZH-Hans');
  });

  it('startsWith chooses the longest prefix branch', async () => {
    const dict = {
      zh: { k: 'ZH' },
      'zh-Hans': { k: 'ZH-Hans' },
    };
    const d = new MiddlewareDispatcher<F>({
      acceptResult: (res) => res != null,
    });
    d.middleware(I18nLookupMiddleware(() => dict, { matchType: 'startsWith' }));

    const a = await d.dispatch('zh-Hans-CN', 'k');
    expect(a).toBe('ZH-Hans');
  });

  it('allows empty string values and treats null/undefined as misses', async () => {
    const dict = {
      en: { a: '', b: null as any, c: undefined as any },
    };
    const d = new MiddlewareDispatcher<F>({
      acceptResult: (res) => res != null,
    });
    d.middleware(I18nLookupMiddleware(() => dict));

    const a = await d.dispatch('en', 'a');
    expect(a).toBe(''); // valid hit

    const b = await d.dispatch('en', 'b');
    const c = await d.dispatch('en', 'c');
    expect(b).toBeUndefined();
    expect(c).toBeUndefined();
  });
});
