import { Awaitable } from './types';

export const patchStringInObject = async <T>(
  obj: T,
  cb: (s: string) => Awaitable<string>,
): Promise<T> => {
  const visited = new WeakSet<object>();

  const isSpecialObject = (v: any): boolean => {
    if (v == null || typeof v !== 'object') return false;
    if (
      v instanceof Date ||
      v instanceof RegExp ||
      v instanceof Map ||
      v instanceof Set ||
      v instanceof WeakMap ||
      v instanceof WeakSet ||
      v instanceof ArrayBuffer ||
      v instanceof DataView ||
      ArrayBuffer.isView(v)
    ) {
      return true;
    }

    const tag = Object.prototype.toString.call(v);
    switch (tag) {
      case '[object URL]':
      case '[object URLSearchParams]':
      case '[object Error]':
      case '[object Blob]':
      case '[object File]':
      case '[object FormData]':
        return true;
      default:
        return false;
    }
  };

  const checkValue = (
    v: any,
  ): { ok: false } | { ok: true; kind: 'string' | 'object' } => {
    if (!v) return { ok: false };

    const t = typeof v;
    if (
      t === 'number' ||
      t === 'bigint' ||
      t === 'symbol' ||
      t === 'function'
    ) {
      return { ok: false };
    }
    if (t === 'string') return { ok: true, kind: 'string' };
    if (t === 'object') return { ok: true, kind: 'object' };

    return { ok: false };
  };

  const rebuildObject = async (value: any): Promise<any> => {
    const proto = Object.getPrototypeOf(value);
    const out = Object.create(proto);

    const keys = Reflect.ownKeys(value);

    await Promise.all(
      keys.map(async (key) => {
        const desc = Object.getOwnPropertyDescriptor(value, key as any);
        if (!desc) return;

        if ('value' in desc) {
          const newVal = await walk(desc.value);
          Object.defineProperty(out, key, { ...desc, value: newVal });
          return;
        }

        Object.defineProperty(out, key, desc);

        let current: any = undefined;
        if (typeof desc.get === 'function') {
          try {
            current = desc.get.call(value);
          } catch {}
        }
        if (current === undefined) return;

        try {
          const newVal = await walk(current);
          if (typeof desc.set === 'function') {
            try {
              desc.set.call(out, newVal);
            } catch {}
          }
        } catch {}
      }),
    );

    return out;
  };

  const walk = async (value: any): Promise<any> => {
    const check = checkValue(value);

    if (!check.ok) return value;

    if (check.kind === 'string') {
      return cb(value);
    }

    if (value instanceof Promise) {
      return value.then((resolved) => walk(resolved));
    }

    if (typeof value === 'object') {
      if (!Array.isArray(value) && isSpecialObject(value)) return value;

      if (visited.has(value)) return value;
      visited.add(value);

      if (Array.isArray(value)) {
        const out = await Promise.all(value.map((v) => walk(v)));
        return out as any;
      }

      return rebuildObject(value);
    }

    return value;
  };

  return walk(obj);
};
