// __tests__/workflow-dispatcher.spec.ts
import { WorkflowDispatcher } from '../src/workflow-dispatcher';

type F = (x: number) => Promise<string>;

function makeSuccess(label: string): F {
  const fn = jest.fn(async (x: number) => `${label}:${x}`);
  return fn as F;
}
function makeAlwaysFail(label: string): F {
  const fn = jest.fn(async () => {
    throw new Error(`fail:${label}`);
  });
  return fn as F;
}
function makeFlaky(label: string, fails: number): F {
  let c = 0;
  const fn = jest.fn(async (x: number) => {
    if (c < fails) {
      c++;
      throw new Error(`flaky-${label}-${c}`);
    }
    return `${label}:${x}`;
  });
  return fn as F;
}
function deferred<T>() {
  let resolve!: (v: T) => void;
  let reject!: (e: any) => void;
  const promise = new Promise<T>((res, rej) => {
    resolve = res;
    reject = rej;
  });
  return { promise, resolve, reject };
}
async function flush(n = 2) {
  for (let i = 0; i < n; i++) await Promise.resolve();
}

describe('WorkflowDispatcher (10ms granularity, no fake timers)', () => {
  test('waits for the first worker to resolve before scheduling', async () => {
    const Adef = deferred<F>(); // pending
    const B = makeSuccess('B'); // active now
    const d = new WorkflowDispatcher<F>([Adef.promise, B], {
      backoffBaseMs: 10,
    });
    const p = d.dispatch(1);
    await flush();
    await expect(p).resolves.toBe('B:1');

    // later A becomes active, then it should be chosen (least-used)
    Adef.resolve(makeSuccess('A'));
    await flush();
    const p2 = d.dispatch(2);
    await expect(p2).resolves.toBe('A:2');
  });

  test('rejects all when all init promises reject', async () => {
    const Adef = deferred<F>(),
      Bdef = deferred<F>();
    const d = new WorkflowDispatcher<F>([Adef.promise, Bdef.promise], {
      backoffBaseMs: 10,
    });
    const p1 = d.dispatch(1);
    const p2 = d.dispatch(2);
    Adef.reject(new Error('A-init-fail'));
    Bdef.reject(new Error('B-init-fail'));
    await flush();
    await expect(p1).rejects.toThrow(/No workers available/);
    await expect(p2).rejects.toThrow(/No workers available/);
  });

  test('dispatch picks the least-used active worker', async () => {
    const A = makeSuccess('A'),
      B = makeSuccess('B');
    const d = new WorkflowDispatcher<F>([A, B], { backoffBaseMs: 10 });
    const r1 = await d.dispatch(1);
    const r2 = await d.dispatch(2);
    const r3 = await d.dispatch(3);
    expect([r1, r2, r3].some((s) => s.startsWith('A'))).toBe(true);
    expect([r1, r2, r3].some((s) => s.startsWith('B'))).toBe(true);
    const actives = d.snapshot().filter((s) => s.status === 'active') as any[];
    expect(actives.reduce((sum, s) => sum + s.totalRuns, 0)).toBe(3);
  });

  test('on failure, it switches workers and throws after all active failed once', async () => {
    const A = makeAlwaysFail('A'),
      B = makeSuccess('B');
    const d = new WorkflowDispatcher<F>([A, B], { backoffBaseMs: 10 });
    await expect(d.dispatch(10)).resolves.toBe('B:10');

    const A2 = makeAlwaysFail('A2'),
      B2 = makeAlwaysFail('B2');
    const d2 = new WorkflowDispatcher<F>([A2, B2], { backoffBaseMs: 10 });
    await expect(d2.dispatch(99)).rejects.toThrow(/fail:(A2|B2)/);
  });

  test('sets backoff and avoids the blocked worker while another is eligible', async () => {
    const A = makeAlwaysFail('A'),
      B = makeSuccess('B');
    const d = new WorkflowDispatcher<F>([A, B], { backoffBaseMs: 10 });

    // First dispatch may fail on A and then succeed elsewhere; we only need a fail to set backoff
    try {
      await d.dispatch(1);
    } catch {
      /* ignore */
    }

    const active = d.snapshot().filter((s) => s.status === 'active') as any[];
    const blocked = active.find((s) => s.failCount > 0);
    if (blocked) {
      expect(blocked.blockedMs).toBeGreaterThanOrEqual(10 - 1); // ~10ms right after failure
      const res = await d.dispatch(2);
      expect(res === 'B:2' || res === 'A:2').toBe(true); // typically B since A is blocked
    }
  });

  test('dispatchSpecific ignores backoff and retries on the same worker (FIFO)', async () => {
    const flaky = makeFlaky('A', 2); // fail, fail, then success
    const B = makeSuccess('B');
    const d = new WorkflowDispatcher<F>([flaky, B], {
      maxAttempts: 3,
      backoffBaseMs: 10,
    });
    const p1 = d.dispatchSpecific(0, 100);
    const p2 = d.dispatchSpecific(0, 200);
    await expect(p1).resolves.toBe('A:100');
    await expect(p2).resolves.toBe('A:200');
    const snap0 = d.snapshot()[0] as any;
    expect(snap0.totalRuns).toBeGreaterThanOrEqual(2);
  });

  test('dispatchSpecific waits for pending worker and fails if its init rejects', async () => {
    const Adef = deferred<F>();
    const B = makeSuccess('B');
    const d = new WorkflowDispatcher<F>([Adef.promise, B], {
      backoffBaseMs: 10,
    });

    // enqueue a specific task to worker 0 (still pending)
    const p = d.dispatchSpecific(0, 1);
    await flush(); // let dispatcher enqueue paths settle

    // Trigger the reject *inside* the expect's promise via an async IIFE.
    await expect(
      (async () => {
        // now reject the init; this happens after expect has attached handlers
        Adef.reject(new Error('A-init-fail'));

        // give the dispatcher a macrotask tick if your impl uses setTimeout(0) to reject
        await new Promise((r) => setTimeout(r, 0));

        // the awaited value for expect(...).rejects is p
        return p;
      })(),
    ).rejects.toThrow(/failed to initialize/i);

    // Calling dispatchSpecific again on the same rejected worker should also reject
    await expect(
      (async () => {
        const p2 = d.dispatchSpecific(0, 2);
        await new Promise((r) => setTimeout(r, 0));
        return p2;
      })(),
    ).rejects.toThrow(/failed to initialize/i);
  });

  test('stops after reaching maxAttempts even if not all active were tried', async () => {
    const A = makeAlwaysFail('A'),
      B = makeAlwaysFail('B');
    const d = new WorkflowDispatcher<F>([A, B], {
      maxAttempts: 2,
      backoffBaseMs: 10,
    });
    await expect(d.dispatch(3)).rejects.toThrow(/fail:(A|B)/);
  });

  test('failCount is decreased after a success (not below zero)', async () => {
    const flaky = makeFlaky('A', 1); // one fail then success
    const B = makeSuccess('B');
    const d = new WorkflowDispatcher<F>([flaky, B], { backoffBaseMs: 10 });
    try {
      await d.dispatch(1);
    } catch {}
    await d.dispatchSpecific(0, 2); // succeed on A; failCount should step down
    const snap0 = d.snapshot()[0] as any;
    expect(snap0.failCount).toBeGreaterThanOrEqual(0);
  });
});
