// tests/dual-object.spec.ts
import { dualizeAny, AsyncMethodKeys } from '../src/dual-object';

type Client = {
  id: string;
  name(): string; // 同步方法
  ping(): Promise<number>; // 异步方法
};

const delay = (ms: number) => new Promise<void>((r) => setTimeout(r, ms));

describe('dualizeAny basic state machine', () => {
  test('sync-first property access => fulfilled (from sync), asyncFn not called', async () => {
    let syncCalled = 0;
    let asyncCalled = 0;

    const obj = dualizeAny<Client>(
      () => {
        syncCalled++;
        return {
          id: 'local',
          name: () => 'cached',
          ping: async () => 1,
        };
      },
      async () => {
        asyncCalled++;
        await delay(30);
        return {
          id: 'remote',
          name: () => 'fresh',
          ping: async () => 42,
        };
      },
      { asyncMethods: ['ping'] satisfies readonly AsyncMethodKeys<Client>[] },
    );

    // undecided 下先读同步属性
    expect(obj.id).toBe('local');
    expect(obj.name()).toBe('cached');
    expect(syncCalled).toBe(1);
    // 现在 await 只会 Promise.resolve(已有值)，不会触发 asyncFn
    const v = await obj;
    expect(v.id).toBe('local');
    expect(asyncCalled).toBe(0);
  });

  test('await-first => pending→fulfilled (from async), sync not called', async () => {
    let syncCalled = 0;
    let asyncCalled = 0;

    const obj = dualizeAny<Client>(
      () => {
        syncCalled++;
        return {
          id: 'local',
          name: () => 'cached',
          ping: async () => 1,
        };
      },
      async () => {
        asyncCalled++;
        await delay(10);
        return {
          id: 'remote',
          name: () => 'fresh',
          ping: async () => 42,
        };
      },
      { asyncMethods: ['ping'] as const },
    );

    // 先 await（或 obj.then），应走 asyncFn
    const v = await obj;
    expect(v.id).toBe('remote');
    expect(syncCalled).toBe(0);
    expect(asyncCalled).toBe(1);

    // fulfilled 后再取属性为同步
    expect(obj.name()).toBe('fresh');
  });

  test('pending: accessing non-async property throws TypeError', async () => {
    const obj = dualizeAny<Client>(
      () => ({
        id: 'local',
        name: () => 'cached',
        ping: async () => 1,
      }),
      async () => {
        await delay(50);
        return {
          id: 'remote',
          name: () => 'fresh',
          ping: async () => 42,
        };
      },
      { asyncMethods: ['ping'] as const },
    );

    // 触发 pending（但不等待完成）
    // 通过 .then 或 await 的方式均可；这里用 .then 触发
    const p = (obj as unknown as Promise<Client>).then(() => {
      /* noop */
    });

    // pending 状态下访问非 asyncMethods 的键应抛错
    expect(() => {
      // eslint-disable-next-line @typescript-eslint/no-unused-expressions
      (obj as Client).id;
    }).toThrow(new TypeError('Value is not ready yet. Please await it first.'));

    await p; // 清理 pending
  });

  test('asyncMethods in undecided: returns deferred function that waits object promise', async () => {
    const obj = dualizeAny<Client>(
      () => ({
        id: 'local',
        name: () => 'cached',
        ping: async () => 1,
      }),
      async () => {
        await delay(20);
        return {
          id: 'remote',
          name: () => 'fresh',
          ping: async () => 42,
        };
      },
      { asyncMethods: ['ping'] as const },
    );

    // undecided 下先访问 async 方法：应进入 pending，并返回一个 Promise<number> 的函数结果
    const r = await obj.ping();
    expect(r).toBe(42);

    // 之后对象应已 fulfilled（来自 async）
    expect(obj.name()).toBe('fresh');
  });

  test('asyncMethods in pending: returns deferred function (no throw)', async () => {
    const obj = dualizeAny<Client>(
      () => ({
        id: 'local',
        name: () => 'cached',
        ping: async () => 1,
      }),
      async () => {
        await delay(30);
        return {
          id: 'remote',
          name: () => 'fresh',
          ping: async () => 42,
        };
      },
      { asyncMethods: ['ping'] as const },
    );

    // 触发 pending
    const start = (obj as unknown as Promise<Client>).then(() => {});
    // pending 下访问 async 方法不抛错，而是返回延迟函数的结果
    const r = await obj.ping();
    expect(r).toBe(42);
    await start;
  });

  test('rejected from sync(): sync() throws => state rejected; any access throws; await rejects', async () => {
    const err = new Error('boom-sync');
    const obj = dualizeAny<Client>(
      () => {
        throw err;
      },
      async () => {
        await delay(10);
        return { id: 'remote', name: () => 'fresh', ping: async () => 42 };
      },
      { asyncMethods: ['ping'] as const },
    );

    // undecided 下访问普通属性会触发 ensureSync → 抛错
    expect(() => (obj as Client).name()).toThrow(err);

    // then/await 也应得到同样的 rejection
    await expect(obj).rejects.toThrow(err);
  });

  test('rejected from async(): await-first triggers async then rejects; further access throws same', async () => {
    const err = new Error('boom-async');
    const obj = dualizeAny<Client>(
      () => ({
        id: 'local',
        name: () => 'cached',
        ping: async () => 1,
      }),
      async () => {
        await delay(10);
        throw err;
      },
      { asyncMethods: ['ping'] as const },
    );

    await expect(obj).rejects.toThrow(err);
    // 之后访问任何键也应抛相同错误
    expect(() => (obj as Client).name()).toThrow(err);
    expect(() => (obj as Client).id).toThrow(err);
  });
});

describe('primitives & coercion', () => {
  test('number primitive: arithmetic and toString/valueOf work in fulfilled (sync)', () => {
    const x = dualizeAny<number>(
      () => 5,
      async () => 99,
    );

    // 首次发生隐式转换，走 sync → fulfilled
    expect(x + 1).toBe(6);
    expect(String(x)).toBe('5');
    // 原型方法（通过装箱）：
    expect((x as any).toFixed(1)).toBe('5.0');
  });

  test('string primitive: template literal, slice', () => {
    const s = dualizeAny<string>(
      () => 'hello',
      async () => 'world',
    );

    expect(`${s} world`).toBe('hello world');
    expect((s as any).slice(0, 3)).toBe('hel');
  });
});

describe('then/catch/finally semantics', () => {
  test('then/catch/finally chaining (fulfilled)', async () => {
    const obj = dualizeAny<{ a: number }>(
      () => ({ a: 1 }),
      async () => ({ a: 2 }),
    );
    // 首次 await 触发 async
    const seen: number[] = [];
    await (obj as unknown as Promise<{ a: number }>)
      .then((v) => {
        seen.push(v.a);
      })
      .finally(() => {
        seen.push(9);
      });

    expect(seen).toEqual([2, 9]);
  });

  test('rejected path via catch (trigger sync first)', async () => {
    const err = new Error('oops');
    const obj = dualizeAny<{ a: number }>(
      () => {
        throw err;
      },
      async () => ({ a: 2 }),
    );

    // 触发同步路径 → 立刻进入 rejected
    expect(() => (obj as any).a).toThrow(err);

    // thenable 现在也应当 rejected
    await expect(obj as unknown as Promise<{ a: number }>).rejects.toBe(err);
  });

  describe('reflect traps: has/ownKeys/getOwnPropertyDescriptor on fulfilled', () => {
    test('has / ownKeys reflect fulfilled value', () => {
      const obj = dualizeAny<{ a: number; b: number }>(
        () => ({ a: 1, b: 2 }),
        async () => ({ a: 10, b: 20 }),
      );

      // 首次同步访问使其 fulfilled(from sync)
      // eslint-disable-next-line @typescript-eslint/no-unused-expressions
      (obj as any).a;

      expect('a' in (obj as any)).toBe(true);
      expect(Object.keys(obj as any).sort()).toEqual(['a', 'b']);
      const desc = Object.getOwnPropertyDescriptor(Object(obj as any), 'a');
      expect(desc?.enumerable).toBe(true);
    });
  });
});
