import { dualizeAny, throwDualPending } from '../dual-object';

type PromiseState<T = any> =
  | { status: 'pending'; value?: undefined; error?: undefined }
  | { status: 'fulfilled'; value: T; error?: undefined }
  | { status: 'rejected'; value?: undefined; error: any };

const promiseStates = new WeakMap<Promise<any>, PromiseState>();

export const isPromiseLike = (value: any): value is Promise<any> =>
  !!value && typeof value.then === 'function';

export const trackPromise = <T>(promise: Promise<T>): PromiseState<T> => {
  const existing = promiseStates.get(promise);
  if (existing) return existing as PromiseState<T>;

  const state = { status: 'pending' } as PromiseState<T>;
  promiseStates.set(promise, state);

  promise.then(
    (value) => {
      (state as any).status = 'fulfilled';
      (state as any).value = value;
    },
    (error) => {
      (state as any).status = 'rejected';
      (state as any).error = error;
    },
  );

  return state;
};

export const wrapMaybePromise = <T>(
  value: T,
  options?: { methodKeys?: Iterable<PropertyKey> },
): T => {
  if (!isPromiseLike(value)) return value;
  const promise = Promise.resolve(value);
  const state = trackPromise(promise);
  if (state.status === 'fulfilled') return state.value;
  if (state.status === 'rejected') throw state.error;
  return dualizeAny<T>(
    () => {
      const current = trackPromise(promise);
      if (current.status === 'fulfilled') return current.value;
      if (current.status === 'rejected') throw current.error;
      throwDualPending();
    },
    () => promise,
    {
      // Intentionally hide strict method return type here.
      asyncMethods: Array.from(options?.methodKeys ?? []) as any,
    },
  );
};

export const createAsyncMethod =
  (inst: any, key: PropertyKey) =>
  (...args: any[]) =>
    Promise.resolve(inst).then((resolved) => {
      const fn = resolved?.[key];
      if (typeof fn !== 'function') {
        throw new TypeError('Target method is not a function');
      }
      return fn.apply(resolved, args);
    });
