import { Awaitable } from './types';

type AnyFunc = (...args: any[]) => any;

type MiddlewareValue<F extends AnyFunc> = Awaited<ReturnType<F>>;
type MiddlewareResult<F extends AnyFunc> = Promise<MiddlewareValue<F>>;
type MiddlewareNext<F extends AnyFunc> = () => MiddlewareResult<F>;
type MiddlewareArgs<F extends AnyFunc> = [
  ...args: Parameters<F>,
  next: MiddlewareNext<F>,
];
type MiddlewareReturn<F extends AnyFunc> = Awaitable<MiddlewareValue<F>>;

export type Middleware<F extends AnyFunc> = (
  ...args: MiddlewareArgs<F>
) => MiddlewareReturn<F>;

export type MiddlewareAcceptResult<F extends AnyFunc> = (
  s: MiddlewareValue<F>,
) => Awaitable<boolean>;
export type MiddlewareErrorHandler<F extends AnyFunc> = (
  e: any,
  args: Parameters<F>,
  next: MiddlewareNext<F>,
) => Awaitable<MiddlewareValue<F>>;

export interface MiddlewareDispatcherOptions<F extends AnyFunc> {
  acceptResult?: MiddlewareAcceptResult<F>;
  errorHandler?: MiddlewareErrorHandler<F>;
}

export class MiddlewareDispatcher<F extends AnyFunc> {
  constructor(private options: MiddlewareDispatcherOptions<F> = {}) {}
  middlewares: Middleware<F>[] = [];

  middleware(mw: Middleware<F>, prior = false) {
    if (prior) {
      this.middlewares.unshift(mw);
    } else {
      this.middlewares.push(mw);
    }
    return this;
  }

  removeMiddleware(mw: Middleware<F>) {
    const index = this.middlewares.indexOf(mw);
    if (index >= 0) {
      this.middlewares.splice(index, 1);
    }
    return this;
  }

  dispatch(...args: Parameters<F>): MiddlewareResult<F> {
    const mws = this.middlewares;
    const acceptResult: MiddlewareAcceptResult<F> =
      this.options.acceptResult || ((res) => res != null);
    const errorHandler: MiddlewareErrorHandler<F> =
      this.options.errorHandler || ((e, args, next) => next());
    const dispatch = async (i: number): MiddlewareResult<F> => {
      if (i >= mws.length) return undefined;

      const mw = mws[i];
      let nextCalled = false;

      const next = async (): MiddlewareResult<F> => {
        if (nextCalled) {
          return undefined;
        }
        nextCalled = true;
        return dispatch(i + 1);
      };

      const runMw = async (cb: () => MiddlewareReturn<F>) => {
        const res = await cb();
        if (!nextCalled && !(await acceptResult(res))) {
          return dispatch(i + 1);
        }
        return res;
      };

      try {
        return await runMw(() => mw(...args, next));
      } catch (e) {
        return await runMw(() => errorHandler(e, args, next));
      }
    };
    return dispatch(0);
  }
}
