import { MiddlewareDispatcher } from '../middleware-dispatcher';
import { parseI18n } from '../utility/parse-i18n';
import { I18nMiddleware, I18nOptions } from './types';
import { patchStringInObject } from '../patch-string-in-object';

export class I18n<Ex extends any[] = []> {
  constructor(private options: I18nOptions) {}

  locales = new Set(this.options.locales);
  defaultLocale = this.options.defaultLocale ?? this.options.locales[0];

  private mw = new MiddlewareDispatcher<
    (locale: string, text: string, ...ex: Ex) => string | undefined
  >({
    acceptResult: (res) => res != null,
    errorHandler: (e) => {
      throw e;
    },
  });

  private shadowMiddlewares: I18nMiddleware<Ex>[] = [];

  middleware(mw: I18nMiddleware<Ex>, prior = false) {
    this.mw.middleware((locale, text, ...args) => {
      const ex = args.slice(0, -1) as Ex;
      const next = args[args.length - 1] as () => Promise<string | undefined>;
      return mw(locale, text, next, ...ex);
    }, prior);
    if (prior) {
      this.shadowMiddlewares.unshift(mw);
    } else {
      this.shadowMiddlewares.push(mw);
    }
    return this;
  }

  removeMiddleware(mw: I18nMiddleware<Ex>) {
    const idx = this.shadowMiddlewares.indexOf(mw);
    if (idx >= 0) {
      this.shadowMiddlewares.splice(idx, 1);
      this.mw.middlewares.splice(idx, 1);
    }
    return this;
  }

  getExactLocale(locale: string) {
    const input = (locale ?? '').trim();
    if (!input) return this.defaultLocale;

    if (this.locales.has(input)) return input;

    // 小写化比较，保留原大小写
    const entries = Array.from(this.locales).map((l) => ({
      orig: l,
      lower: l.toLowerCase(),
    }));
    const lower = input.toLowerCase();

    // 1) 精确匹配（大小写不敏感）
    const exact = entries.find((e) => e.lower === lower);
    if (exact) return exact.orig;

    // 2) 按 '-' 拆分，依次尝试去掉最右边的段
    //    zh-Hans-CN → zh-Hans → zh
    const parts = lower.split('-');
    while (parts.length > 1) {
      parts.pop();
      const candidate = parts.join('-');
      const hit = entries.find((e) => e.lower === candidate);
      if (hit) return hit.orig;
    }

    // 3) 兜底
    return this.defaultLocale;
  }

  private buildFallbackChain(locale: string): string[] {
    const best = this.getExactLocale(locale); // 你的“最长匹配”函数
    // 拆分 zh-Hans-CN -> ['zh-Hans-CN','zh-Hans','zh']
    const parts: string[] = [];
    const segs = best.split('-');
    for (let i = segs.length; i > 1; i--)
      parts.push(segs.slice(0, i).join('-'));
    parts.push(segs[0]); // 'zh'
    // 附加默认语言
    if (!parts.includes(this.defaultLocale)) parts.push(this.defaultLocale);
    // 去重
    return Array.from(new Set(parts)).filter((p) => this.locales.has(p));
  }

  private async applyMiddlewares(
    locale: string,
    text: string,
    ...ex: Ex
  ): Promise<string | undefined> {
    const tryLocale = (locale: string) => this.mw.dispatch(locale, text, ...ex);

    for (const loc of this.buildFallbackChain(locale)) {
      const result = await tryLocale(loc);
      if (result != null) {
        return result;
      }
    }
    return undefined;
  }

  async translateString(
    locale: string,
    text: string,
    ...ex: Ex
  ): Promise<string> {
    if (!text) return text;

    locale = this.getExactLocale(locale);

    const pieces = parseI18n(text);
    if (!pieces.some((p) => p.type === 'ph')) {
      return pieces
        .map((p) => (p.type === 'raw' ? p.value : `#{${p.rawInner}}`))
        .join('');
    }

    const promises: Array<Promise<string | undefined | null>> = [];

    for (const p of pieces) {
      if (p.type === 'ph') {
        promises.push(this.applyMiddlewares(locale, p.key, ...ex));
      }
    }

    const results = await Promise.all(promises);

    let out = '';
    let k = 0;
    for (const p of pieces) {
      if (p.type === 'raw') {
        out += p.value;
      } else {
        const r = results[k++];
        out += r == null ? `#{${p.rawInner}}` : r;
      }
    }
    return out;
  }

  async translate<T>(locale: string, obj: T, ...ex: Ex): Promise<T> {
    return patchStringInObject(obj, (s) =>
      this.translateString(locale, s, ...ex),
    );
  }
}
