import { workflow } from '../src/workflow';

const sleep = (ms: number) => new Promise((res) => setTimeout(res, ms));

describe('workflow – basics', () => {
  it('mixes async + sync + property + invoke (() ) correctly', async () => {
    class A {
      value = 42;

      async foo() {
        await sleep(5);
        this.value += 42; // 84
        return this;
      }

      bar() {
        this.value += 10; // 94
        return this;
      }

      baz() {
        // 返回可调用：再调用一次时把 value+15 然后返回 this
        return () => {
          this.value += 15; // 109
          return this;
        };
      }
    }

    const a = new A();
    const result = await workflow(a).foo().bar().baz()().value;
    await expect(result).toBe(109);
  });

  it('property-only terminal await works', async () => {
    class A {
      value = 7;
    }
    const a = new A();
    const result = await workflow(a).value;
    await expect(result).toBe(7);
  });

  it('throws when invoking a non-function value', async () => {
    class A {
      value = 123;
    }
    const a = new A();
    // @ts-expect-error intentional wrong call
    await expect(workflow(a).value()).rejects.toThrow(TypeError);
  });

  it('throws when calling a non-existing method', async () => {
    class A {
      ok() {
        return 1;
      }
    }
    const a: any = new A();
    // @ts-expect-error intentional wrong call
    await expect(workflow(a).noSuchMethod()).rejects.toThrow();
  });

  it('preserves "this" binding', async () => {
    class C {
      x = 1;
      inc(by = 1) {
        this.x += by;
        return this;
      }
      async incAsync(by = 1) {
        await sleep(1);
        this.x += by;
        return this;
      }
    }
    const c = new C();
    await workflow(c).inc(2).incAsync(3);
    expect(c.x).toBe(1 + 2 + 3);
  });
});

describe('workflow – memoization of execution (multi then/catch/finally)', () => {
  it('executes only once even with multiple then()', async () => {
    const calls: string[] = [];
    class A {
      async foo() {
        calls.push('foo');
        await sleep(2);
        return this;
      }
      bar() {
        calls.push('bar');
        return this;
      }
    }
    const chain = workflow(new A()).foo().bar();

    const p1 = chain.then((v) => v);
    const p2 = chain.then((v) => v);
    const [r1, r2] = await Promise.all([p1, p2]); // 不应重复执行链

    expect(r1).toBe(r2);
    expect(calls.join(',')).toBe('foo,bar'); // 各一步只一次
  });

  it('catch/finally also reuse the same promise', async () => {
    const steps: string[] = [];
    class A {
      async bad() {
        steps.push('bad');
        await sleep(1);
        throw new Error('boom');
      }
    }
    const chain = workflow(new A()).bad();

    const p1 = chain.catch(() => {
      steps.push('c1');
      return 'C1';
    });
    const p2 = chain.finally(() => {
      steps.push('f1');
    });

    const [r1] = await Promise.allSettled([p1, p2]);
    // bad 只执行一次；catch/finally 都被触发
    expect(steps).toEqual(['bad', 'c1', 'f1']);
    expect(r1.status).toBe('fulfilled');
  });
});

describe('workflow – function value invocation foo()()', () => {
  it('supports invoking current value (invoke step)', async () => {
    class F {
      n = 0;
      makeAdder(start: number) {
        this.n += start; // 先加 start
        return (inc: number) => {
          // 再可调用一次
          this.n += inc;
          return this;
        };
      }
    }
    const f = new F();
    await workflow(f).makeAdder(5)(7);
    expect(f.n).toBe(12);
  });
});

describe('workflow – branching / prefix reuse', () => {
  it('reuses prefix (like connected client) and branches run independently', async () => {
    const log: string[] = [];

    class Client {
      connects = 0;
      qs = 0;
      gets = 0;
      posts = 0;

      async connect() {
        this.connects++;
        log.push('connect');
        await sleep(3);
        return this;
      }
      query() {
        this.qs++;
        log.push('query');
        return this;
      }
      async get() {
        this.gets++;
        log.push('get');
        await sleep(1);
        return 'GET';
      }
      async post() {
        this.posts++;
        log.push('post');
        await sleep(1);
        return 'POST';
      }
    }

    const client = workflow(new Client()).connect();

    // 两个分支从同一前缀出发
    const p1 = client.query().get();
    const p2 = client.post();

    const [r1, r2] = await Promise.all([p1, p2]);
    expect(r1).toBe('GET');
    expect(r2).toBe('POST');

    // 前缀 connect 只跑一次；各分支自己的步骤各一次
    expect(log.filter((x) => x === 'connect').length).toBe(1);
    expect(log.filter((x) => x === 'query').length).toBe(1);
    expect(log.filter((x) => x === 'get').length).toBe(1);
    expect(log.filter((x) => x === 'post').length).toBe(1);
  });

  it('parallel awaits still reuse the same prefix result', async () => {
    const counts = { connect: 0, get: 0, post: 0 };
    class Client {
      async connect() {
        counts.connect++;
        await sleep(2);
        return this;
      }
      async get() {
        counts.get++;
        await sleep(1);
        return 1;
      }
      async post() {
        counts.post++;
        await sleep(1);
        return 2;
      }
    }
    const client = workflow(new Client()).connect();
    const [a, b] = await Promise.all([client.get(), client.post()]);
    expect(a + b).toBe(3);
    expect(counts).toEqual({ connect: 1, get: 1, post: 1 });
  });
});

describe('workflow – property chains', () => {
  it('handles deep property then method: .bar.baz(2,3)', async () => {
    class B {
      bazCalls = 0;
      baz(u: number, v: number) {
        this.bazCalls++;
        return u + v;
      }
    }
    class A {
      bar = new B();
    }

    const sum = await workflow(new A()).bar.baz(4, 5);
    expect(sum).toBe(9);
  });

  it('deep property terminal await returns value', async () => {
    class A {
      cfg = { feature: { enabled: true as boolean } };
    }
    const result = await workflow(new A()).cfg.feature.enabled;
    await expect(result).toBe(true);
  });
});

describe('workflow – error propagation', () => {
  it('propagates thrown sync errors', async () => {
    class A {
      bad() {
        throw new Error('sync boom');
      }
    }
    await expect(workflow(new A()).bad()).rejects.toThrow('sync boom');
  });

  it('propagates rejected async errors', async () => {
    class A {
      async bad() {
        await sleep(1);
        throw new Error('async boom');
      }
    }
    await expect(workflow(new A()).bad()).rejects.toThrow('async boom');
  });
});

describe('workflow – order and once-only semantics in longer chains', () => {
  it('runs steps in strict order and exactly once each', async () => {
    const order: string[] = [];
    class A {
      async a() {
        order.push('a');
        await sleep(1);
        return this;
      }
      b() {
        order.push('b');
        return this;
      }
      async c() {
        order.push('c');
        await sleep(1);
        return this;
      }
      d() {
        order.push('d');
        return 123;
      }
    }

    const chain = workflow(new A()).a().b().c().d();
    const p1 = chain.then((x) => x);
    const p2 = chain.then((x) => x);
    const [r1, r2] = await Promise.all([p1, p2]);

    expect(r1).toBe(123);
    expect(r2).toBe(123);
    expect(order).toEqual(['a', 'b', 'c', 'd']); // 没有重复
  });
});

describe('workflow – Promise combinators interop', () => {
  class Client {
    counts = { connect: 0, get: 0, fail: 0 };

    async connect() {
      this.counts.connect++;
      await sleep(5);
      return this;
    }
    async get(val: any, ms = 10) {
      this.counts.get++;
      await sleep(ms);
      return val;
    }
    async fail(msg = 'boom', ms = 10) {
      this.counts.fail++;
      await sleep(ms);
      throw new Error(msg);
    }
  }

  it('works with Promise.all: parallel branches reuse prefix once', async () => {
    const c = new Client();
    const client = workflow(c).connect();

    const [a, b, cval] = await Promise.all([
      client.get(1, 15), // 慢一些
      client.get(2, 5), // 快一些
      client.get(3, 10),
    ]);

    expect([a, b, cval]).toEqual([1, 2, 3]);
    expect(c.counts).toEqual({ connect: 1, get: 3, fail: 0 }); // 前缀 connect 只执行一次
  });

  it('works with Promise.allSettled: successes and failures settle together', async () => {
    const c = new Client();
    const client = workflow(c).connect();

    const settled = await Promise.allSettled([
      client.get('ok-1', 5),
      client.fail('bad-1', 8),
      client.get('ok-2', 2),
      client.fail('bad-2', 1),
    ]);

    // 校验形状和顺序
    expect(settled.map((s) => s.status)).toEqual([
      'fulfilled',
      'rejected',
      'fulfilled',
      'rejected',
    ]);
    const msgs = settled.map(
      (s) => (s as any).reason?.message ?? (s as any).value,
    );
    expect(msgs).toEqual(['ok-1', 'bad-1', 'ok-2', 'bad-2']);

    expect(c.counts.connect).toBe(1); // 前缀仍只连一次
    expect(c.counts.get).toBe(2);
    expect(c.counts.fail).toBe(2);
  });

  it('works with Promise.any: resolves to first fulfilled value even if others reject', async () => {
    const c = new Client();
    const client = workflow(c).connect();

    // 一个快速 reject，一个稍慢 resolve，一个更慢 reject
    const val = await Promise.any([
      client.fail('bad-fast', 2),
      client.get('good', 5),
      client.fail('bad-slow', 20),
    ]);

    expect(val).toBe('good');
    expect(c.counts.connect).toBe(1);
  });

  it('Promise.any rejects with AggregateError only if all reject', async () => {
    const c = new Client();
    const client = workflow(c).connect();

    await expect(
      Promise.any([
        client.fail('e1', 3),
        client.fail('e2', 1),
        client.fail('e3', 2),
      ]),
    ).rejects.toHaveProperty('errors'); // AggregateError.errors 存在
  });

  it('works with Promise.race: resolves/rejects with first settled branch', async () => {
    const c = new Client();
    const client = workflow(c).connect();

    // 先测试最快 resolve 赢
    const r1 = await Promise.race([
      client.get('win', 3),
      client.get('lose', 10),
      client.fail('should-not-win', 2_000),
    ]);
    expect(r1).toBe('win');

    // 再测试最快 reject 赢
    await expect(
      Promise.race([client.fail('first-error', 4), client.get('slow-ok', 20)]),
    ).rejects.toThrow('first-error');

    expect(c.counts.connect).toBe(1); // 两次 race 都复用同一前缀
  });

  it('combinators do not cause duplicate execution on the same branch (memoized)', async () => {
    const c = new Client();
    const branch = workflow(c).connect().get('once', 5);

    const [x, y, z] = await Promise.all([branch, branch, branch]);
    expect(x).toBe('once');
    expect(y).toBe('once');
    expect(z).toBe('once');

    // connect 一次；get 这条分支只计算一次
    expect(c.counts.connect).toBe(1);
    expect(c.counts.get).toBe(1);
  });
});

describe('workflow – nested workflow(workflow(x)) semantics', () => {
  class Svc {
    count = 0;
    async inc(by = 1, ms = 2) {
      await sleep(ms);
      this.count += by;
      return this;
    }
    val() {
      return this.count;
    }
  }

  it('is idempotent: wrapping an existing chain returns the same chain (no new steps)', async () => {
    const s = new Svc();

    const c1 = workflow(s).inc(2); // 构建一条链
    const c2 = workflow(c1); // 套娃：应当恒等
    expect(c2).toBe(c1); // 引用相等（若你选择“同 Node 新代理”，可改为 not.toBe 但 Node 相等）

    // 两次 then 也只执行一遍链
    const [a, b] = await Promise.all([c1.then((x) => x), c2.then((x) => x)]);
    expect(a).toBe(b);
    expect(s.count).toBe(2);
  });

  it('does not double-execute when awaited via inner and outer chains', async () => {
    const s = new Svc();
    const inner = workflow(s).inc(3);
    const outer = workflow(inner); // 恒等

    await inner;
    await outer; // 不应重复执行
    expect(s.count).toBe(3);
  });

  it('branches share the same instance; results are {6,8} and final count is 8', async () => {
    const s = new Svc();

    // 前缀（未执行前只是定义步骤）
    const base = workflow(s).inc(5, 1); // -> count: 5
    const wrapped = workflow(base); // 恒等返回

    // 分支一：先 +1（较快完成）
    const b1 = wrapped.inc(1, 5).val(); // 5 -> 6
    // 分支二：后 +2（稍慢完成）
    const b2 = wrapped.inc(2, 10).val(); // 6 -> 8 （若它先完成，则 5->7，再被另一条改成 8）

    const [v1, v2] = await Promise.all([b1, b2]);

    // 两个返回值是 6 和 8（顺序不保证）
    expect(new Set([v1, v2])).toEqual(new Set([6, 8]));
    // 最终状态必然为 8
    expect(s.count).toBe(8);
  });

  it('interop with Promise combinators remains correct when nested', async () => {
    const s = new Svc();
    const chain = workflow(s).inc(1); // 前缀
    const nested = workflow(chain); // 恒等

    const [all, race, any] = await Promise.all([
      Promise.all([nested.val(), nested.inc(1).val()]), // [1, 2]
      Promise.race([nested.val(), nested.inc(1).val()]), // 先 settle 的可能是 val()（1）
      Promise.any([nested.val(), nested.inc(1).val()]), // 第一个 fulfill
    ]);

    expect(all).toEqual([1, 2]);
    expect([race === 1, race === 2].some(Boolean)).toBe(true);
    expect([any === 1, any === 2].some(Boolean)).toBe(true);
  });
});

describe('workflow – wrapping a method chain property', () => {
  class A {
    value = 0;

    async foo() {
      await sleep(2);
      this.value += 10;
      return this;
    }

    async bar() {
      await sleep(2);
      this.value += 5;
      return this;
    }
  }

  it('workflow(workflow(a).foo)() should behave like workflow(a).foo()', async () => {
    const a1 = new A();
    const a2 = new A();

    // baseline：直接调用 workflow(a).foo()
    await workflow(a1).foo();
    const baselineValue = a1.value;

    // wrapped：先拿出 workflow(a).foo 再 workflow 一次
    const wrapped = workflow(workflow(a2).foo);
    await wrapped(); // 调用它

    const wrappedValue = a2.value;

    // 两者行为应该一致
    expect(wrappedValue).toBe(baselineValue);
    expect(wrappedValue).toBe(10);
  });

  it('should also support chaining after such wrapped call', async () => {
    const a = new A();
    // workflow(workflow(a).foo)() 之后还能继续链
    const result = await workflow(workflow(a).foo)().bar();
    expect(result.value).toBe(15); // foo +10, bar +5
  });
});
