import { AnyClass, Empty } from '../types';
import {
  AppContext,
  AppServiceClass,
  AppProvideArgs,
  AppProvidedMerged,
  AppProvideOptions,
  AppContextUsed,
} from './types';
import {
  createAsyncMethod,
  isPromiseLike,
  trackPromise,
  wrapMaybePromise,
} from './promise-utils';

const ProvidePrefix = 'provide:';

const getMethodDescriptor = (cls: AnyClass, key: PropertyKey) => {
  let proto = cls.prototype;
  while (proto && proto !== Object.prototype) {
    const desc = Object.getOwnPropertyDescriptor(proto, key as any);
    if (desc) return desc;
    proto = Object.getPrototypeOf(proto);
  }
  return undefined;
};

type LoadEntry = {
  classRef: AnyClass;
  inst: any;
  methodKeys: Set<PropertyKey>;
  pendingSets: Array<{ key: PropertyKey; value: any }>;
};

type ObjectStep = (ctx: AppContextCore<any, any>) => void;

const flushPendingSets = (entry: LoadEntry) => {
  if (isPromiseLike(entry.inst)) return;
  if (!entry.pendingSets.length) return;
  for (const item of entry.pendingSets) {
    (entry.inst as any)[item.key] = item.value;
  }
  entry.pendingSets.length = 0;
};

const resolveEntryIfNeeded = async (entry: LoadEntry) => {
  if (isPromiseLike(entry.inst)) {
    entry.inst = await entry.inst;
  }
  flushPendingSets(entry);
  return entry.inst;
};

export class AppContextCore<Cur = Empty, Req = Empty> {
  private __current: Cur;
  private __required: Req;

  registry = new Map<string | AnyClass, LoadEntry>();
  loadSeq: LoadEntry[] = [];
  objectSteps: ObjectStep[] = [];

  provide<
    C extends AppServiceClass<Cur, Req>,
    const P extends string = '',
    const M extends (keyof InstanceType<C>)[] = [],
  >(
    cls: C,
    ...args: AppProvideArgs<Cur, Req, C, P, M>
  ): AppProvidedMerged<Cur, Req, C, P, M> {
    const last = args[args.length - 1] as any;
    const hasOptions =
      !!last &&
      typeof last === 'object' &&
      ('provide' in last ||
        'merge' in last ||
        'useValue' in last ||
        'useFactory' in last ||
        'useClass' in last);

    const options = (hasOptions ? last : undefined) as
      | AppProvideOptions<Cur, Req, C, P, M>
      | undefined;

    const _args = (
      hasOptions ? args.slice(0, -1) : args
    ) as ConstructorParameters<C>;

    const inst =
      options?.useValue ??
      (options?.useFactory
        ? options.useFactory(this as any, ..._args)
        : new (options?.useClass ?? cls)(this as any, ..._args));

    const classRef = cls as unknown as AnyClass;
    const provideKey = options?.provide
      ? ProvidePrefix + String(options.provide)
      : undefined;

    const methodKeys = new Set<PropertyKey>();
    for (const name of Object.getOwnPropertyNames(cls.prototype)) {
      if (name === 'constructor') continue;
      const desc = Object.getOwnPropertyDescriptor(cls.prototype, name);
      if (desc && typeof desc.value === 'function') {
        methodKeys.add(name);
      }
    }

    const entry: LoadEntry = {
      classRef,
      inst,
      methodKeys,
      pendingSets: [],
    };
    this.registry.set(classRef, entry);
    if (provideKey) this.registry.set(provideKey, entry);
    this.loadSeq.push(entry);

    if (options?.provide) {
      const prop = options.provide;
      const step: ObjectStep = (ctx) => {
        Object.defineProperty(ctx, prop, {
          enumerable: true,
          configurable: true,
          get: () => {
            const currentEntry = ctx.registry.get(classRef);
            return wrapMaybePromise(currentEntry?.inst, { methodKeys });
          },
        });
      };
      step(this);
      this.objectSteps.push(step);
    }

    if (options?.merge?.length) {
      for (const key of options.merge) {
        const desc = getMethodDescriptor(cls, key);
        const isMethod = !!desc && typeof desc.value === 'function';

        const step: ObjectStep = (ctx) => {
          if (isMethod) {
            Object.defineProperty(ctx, key, {
              enumerable: true,
              configurable: true,
              get: () => {
                const currentEntry = ctx.registry.get(classRef);
                const currentInst = currentEntry?.inst;
                if (isPromiseLike(currentInst))
                  return createAsyncMethod(currentInst, key);
                const fn = (currentInst as any)?.[key];
                return typeof fn === 'function' ? fn.bind(currentInst) : fn;
              },
            });
            return;
          }

          Object.defineProperty(ctx, key, {
            enumerable: true,
            configurable: true,
            get: () => {
              const currentEntry = ctx.registry.get(classRef);
              const target = wrapMaybePromise(currentEntry?.inst);
              return (target as any)?.[key];
            },
            set: (value: any) => {
              const currentEntry = ctx.registry.get(classRef);
              if (!currentEntry) return;
              if (!isPromiseLike(currentEntry.inst)) {
                (currentEntry.inst as any)[key] = value;
                return;
              }
              const state = trackPromise(currentEntry.inst);
              if (state.status === 'fulfilled') {
                currentEntry.inst = state.value;
                flushPendingSets(currentEntry);
                (currentEntry.inst as any)[key] = value;
                return;
              }
              if (state.status === 'rejected') throw state.error;
              currentEntry.pendingSets.push({ key, value });
            },
          });
        };
        step(this);
        this.objectSteps.push(step);
      }
    }

    return this as any;
  }

  get<R>(cls: AppServiceClass<Cur, Req, any, R>): R {
    const key = cls as unknown as AnyClass;
    if (!this.registry.has(key)) {
      throw new Error(`Service not provided: ${cls.name}`);
    }
    const entry = this.registry.get(key)!;
    const inst = entry.inst;
    if (isPromiseLike(inst)) {
      const state = trackPromise(inst);
      if (state.status === 'fulfilled') {
        entry.inst = state.value;
        flushPendingSets(entry);
        return state.value as any;
      }
      if (state.status === 'rejected') throw state.error;
      return wrapMaybePromise(inst, {
        methodKeys: entry.methodKeys,
      }) as any;
    }
    flushPendingSets(entry);
    return inst as any;
  }

  async getAsync<R>(cls: AppServiceClass<Cur, Req, any, R>): Promise<R> {
    const key = cls as unknown as AnyClass;
    if (!this.registry.has(key)) {
      throw new Error(`Service not provided: ${cls.name}`);
    }
    const entry = this.registry.get(key)!;
    const resolved = await resolveEntryIfNeeded(entry);
    return resolved as R;
  }

  use<const Ctxes extends AppContext<any, any>[]>(
    ...ctxes: Ctxes
  ): AppContextUsed<Cur, Req, Ctxes> {
    for (const ctx of ctxes) {
      const other = ctx as any as AppContextCore<any, any>;
      const entryMap = new Map<LoadEntry, LoadEntry>();

      if (Array.isArray(other?.loadSeq)) {
        const copiedSeq = other.loadSeq.map((item) => ({
          ...item,
          methodKeys: new Set(item.methodKeys ?? []),
          pendingSets: [...(item.pendingSets ?? [])],
        }));
        other.loadSeq.forEach((item, index) => {
          entryMap.set(item, copiedSeq[index]);
        });
        this.loadSeq.push(...copiedSeq);
      }

      if (other?.registry instanceof Map) {
        for (const [key, value] of other.registry.entries()) {
          this.registry.set(key, entryMap.get(value) ?? value);
        }
      }

      if (Array.isArray(other?.objectSteps)) {
        this.objectSteps.push(...other.objectSteps);
        for (const step of other.objectSteps) {
          step(this);
        }
      }
    }

    return this as any;
  }

  define(): AppContext<Cur, Req> {
    return this as any;
  }

  async start(): Promise<Empty extends Req ? AppContext<Cur, Req> : never> {
    for (const entry of this.loadSeq) {
      await resolveEntryIfNeeded(entry);
    }

    for (const entry of this.loadSeq) {
      const inst = entry.inst;
      if (inst && typeof inst.init === 'function') {
        await inst.init();
      }
    }

    return this as any;
  }
}

export const createAppContext = <Req = Empty>() =>
  new AppContextCore<Empty, Req>() as AppContext<Empty, Req>;

// testing code below

class Foo {
  constructor(
    public ctx: AppContext,
    private foo = 3,
  ) {}

  getFoo() {
    return this.foo;
  }
}

class Bar {
  constructor(
    public ctx: AppContext<{ foo: Foo }>,
    private bar: number,
  ) {}

  getBar() {
    return this.bar + this.ctx.foo.getFoo();
  }
}

class Baz {
  constructor(public ctx: AppContext<{ foo: Foo }>) {}

  getBaz() {
    return this.ctx.foo.getFoo() * 2;
  }
}

async function test() {
  const ctx1 = createAppContext()
    .provide(Foo, 3, { provide: 'foo' })
    .provide(Bar, 5, {
      merge: ['getBar'],
    })
    .define();

  const ctx2 = createAppContext<{ foo: Foo }>()
    .provide(Baz, {
      provide: 'baz',
    })
    .define();

  const ctx2Used = await createAppContext().use(ctx2).define().start();
  const final = await createAppContext().use(ctx1).use(ctx2).define().start();
  const reversed = await createAppContext()
    .use(ctx2)
    .use(ctx1)
    .define()
    .start();
}
