import fs from 'fs';
import { randomEnglish, randomUUID } from 'make-random';

export interface Ping {
  version: Version;
  players: Players;
  description: Description;
  favicon: string;
}

export interface Description {
  text: string;
}

export interface Players {
  max: number;
  online: number;
  sample: Sample[];
}

export interface Sample {
  name: string;
  id: string;
}

export interface Version {
  name: string;
  protocol: number;
}

export class PingGenerator {
  private icon: string;
  private players: Players;
  private max = 20;
  private currentMax = 10;
  private currentMin = 5;

  private refreshPeriod = 100;
  private pointer = 0;

  async refreshPlayers() {
    this.pointer = 0;
    this.players = {
      max: this.max,
      online:
        Math.floor(Math.random() * (this.currentMax - this.currentMin + 1)) +
        this.currentMin,
      sample: [
        {
          name: await randomEnglish(1),
          id: await randomUUID(),
        },
      ],
    };
  }

  async getPlayers() {
    if (!this.players || ++this.pointer >= this.refreshPeriod) {
      await this.refreshPlayers();
    }
    return this.players;
  }

  async generate(protocol: number): Promise<Ping> {
    return {
      version: {
        name: process.env.MC_VERSION || '1.12.2',
        protocol,
      },
      players: await this.getPlayers(),
      description: {
        text: process.env.MC_MOTD || 'A Minecraft server',
      },
      favicon: this.icon ? `data:image/png;base64,${this.icon}` : undefined,
    };
  }

  load;

  constructor() {
    if (process.env.MC_MAX) {
      this.max = parseInt(process.env.MC_MAX);
    }
    if (process.env.MC_CURRENT_MAX) {
      this.currentMax = parseInt(process.env.MC_CURRENT_MAX);
    }
    if (process.env.MC_CURRENT_MIN) {
      this.currentMin = parseInt(process.env.MC_CURRENT_MIN);
    }
    if (process.env.MC_REFRESH_COUNT) {
      this.refreshPeriod = parseInt(process.env.MC_REFRESH_COUNT);
    }
    fs.promises
      .readFile('./icon.png', 'base64')
      .then((data) => {
        this.icon = data;
      })
      .catch();
  }
  kickMessage(name: string) {
    const message =
      process.env.MC_KICK_MESSAGE ||
      'You ({name}) are not allowed to join this server.';
    return message.replaceAll('{name}', name);
  }
}
