import net from 'net';
import { createLogger } from 'bunyan';
import { parsePacket, encode } from './src/decode';
import { PingGenerator } from './src/ping';

const server = net.createServer();
const log = createLogger({ name: 'minecraft-stub' });
const generator = new PingGenerator();

server.on('error', (err) => {
  log.error('Server error', err);
});

server.on('connection', (socket) => {
  let buffer = Buffer.alloc(0);
  socket.on('data', async (data) => {
    buffer = Buffer.concat([buffer, data]);
    const baseBuffer = buffer.slice(buffer.indexOf(0x00) + 1, buffer.length);
    const baseHostname =
      parsePacket.hostname(baseBuffer).length <= 10
        ? parsePacket.hostname(baseBuffer).length + 10
        : parsePacket.hostname(baseBuffer).length;
    if (buffer.length < baseHostname && buffer.length > 10) return;
    try {
      const player = parsePacket.player(buffer, baseBuffer);
      const protocol = parsePacket.protocol(baseBuffer);
      const hostname = parsePacket.hostname(baseBuffer);
      const port = parsePacket.port(baseBuffer);
      const state = parsePacket.state(baseBuffer);

      // Perhaps someone can change the logging message? :/
      switch (state) {
        case 1:
          /*
          Response handshake with server status
          https://wiki.vg/Server_List_Ping#Handshake
          */

          log.info(
            `Handshake is initiated with hostname ${hostname} and port ${port}`,
          );

          const config = await generator.generate(protocol);

          socket.write(encode(config));
          break;
        case 2:
          /*
          Response when client tries to connect
          https://wiki.vg/Protocol#Disconnect_.28login.29
          */

          log.info(`${player} is trying to connect to ${hostname}:${port}`);

          socket.write(encode(generator.kickMessage(player)));
          break;
        default:
          log.info(`Responding to ping`);
          socket.write(buffer);
          break;
      }
    } catch (error) {
      log.error(error);
    } finally {
      buffer = Buffer.alloc(0);
    }
  });
});

const port = parseInt(process.env.PORT || '25565', 10);

server.listen(port, null, () => {
  log.info(`Fake Minecraft Server Listening in ${port}`);
});
