import fs from 'fs';
import moment from 'moment';
import jinja from 'jinja-js';
import yaml from 'yaml';
import _ from 'lodash';

interface Member {
  name: string;
  qq: number;
}
interface Team {
  name: string;
  members: Member[];
}
interface Match {
  team1: string;
  team2: string;
}
interface Config {
  template: string;
  matches: Match[];
  teams: Team[];
}
interface RenderData {
  teamNames: string[];
  timeString: string;
  duels: Member[][];
}

const config: Config = yaml.parse(
  fs.readFileSync(process.argv[2] || './config.yaml', 'utf-8'),
);

function shuffle<T>(arr: T[]): T[] {
  for (let i = arr.length - 1; i >= 0; i--) {
    const rIndex = Math.floor(Math.random() * (i + 1));
    const temp = arr[rIndex];
    arr[rIndex] = arr[i];
    arr[i] = temp;
  }
  return arr;
}

function makeMatch(team1: Team, team2: Team): RenderData {
  if (team1.members.length !== team2.members.length) {
    throw `Team ${team1.name} and ${team2.name} has different member size.`;
  }
  const len = team1.members.length;
  const memberLists: Member[][] = [team1, team2].map((team) => {
    const list = _.clone(team.members);
    shuffle(list);
    return list;
  });
  const duels: Member[][] = [];
  for (let i = 0; i < len; ++i) {
    duels.push([memberLists[0][i], memberLists[1][i]]);
  }
  return {
    teamNames: [team1.name, team2.name].map((n) => n.toUpperCase()),
    timeString: moment().format('YYYY.MM.DD'),
    duels,
  };
}
const missingTeams: string[] = [];
for (const match of config.matches) {
  const team1 = _.find(
    config.teams,
    (t) => t.name.toUpperCase() === match.team1.toUpperCase(),
  );
  if (!team1) {
    missingTeams.push(match.team1);
  }
  const team2 = _.find(
    config.teams,
    (t) => t.name.toUpperCase() === match.team2.toUpperCase(),
  );
  if (!team2) {
    missingTeams.push(match.team2);
  }
  if (!team1 || !team2) {
    continue;
  }
  const data = makeMatch(team1, team2);
  const renderedData = jinja.render(config.template, data);
  console.log(renderedData);
  console.log('');
}
for (const missingName of missingTeams) {
  console.error(`Missing team ${missingName.toUpperCase()}.`);
}
