import { Bot, Context, Logger, Session } from 'koishi';
import { GuildVerifierConfig, GuildVerifierConfigLike } from './config';
import {
  DefinePlugin,
  InjectConfig,
  OnPlatform,
  UseEvent,
  OnPrivate,
  UseCommand,
  CommandUsage,
  PutSession,
  PutArg,
  InjectLogger,
  CommandDescription,
} from 'koishi-thirdeye';
export * from './config';

@OnPlatform('onebot')
@DefinePlugin({ name: 'guild-verifier', schema: GuildVerifierConfig })
export default class GuildVerifier {
  constructor(private ctx: Context, config: GuildVerifierConfigLike) {}

  @InjectConfig()
  private config: GuildVerifierConfig;

  private getUserDisplayName(session: Session) {
    return session.userId;
  }

  private getGuildDisplayName(session: Session) {
    return session.guildId;
  }

  @UseEvent('guild-request')
  async onGuildRequest(session: Session) {
    const userDisplayName = this.getUserDisplayName(session);
    const guildDisplayName = this.getGuildDisplayName(session);
    this.logger.info(
      `${session.messageId}: ${userDisplayName} invited ${session.selfId} to ${guildDisplayName}.`,
    );
    if (session.userId && this.config.isAdmin(session.userId)) {
      this.logger.info(
        `${session.messageId}: Accepted because ${userDisplayName} is an admin.`,
      );
      try {
        await session.bot.handleGuildRequest(session.messageId, true);
      } catch (e) {
        this.logger.error(
          `handleGuildRequest for ${session.messageId} failed: ${e.toString()}`,
        );
      }
      return;
    }
    if (!this.config.notifyGuildId) {
      return;
    }
    await session.bot.sendMessage(
      this.config.notifyGuildId,
      `${userDisplayName} 邀请我加入群 ${guildDisplayName} 。\n请私聊发送我 verify.accept ${session.messageId} 同意。\n请私聊发送我 verify.reject ${session.messageId} 拒绝。`,
    );
    return;
  }

  @OnPrivate()
  @UseCommand('verify', '处理邀请入群请求', { empty: true })
  @CommandDescription({ en: 'Handle guild invitation request' })
  verifyCommand() {}

  @InjectLogger()
  private logger: Logger;

  private async handleRequest(
    session: Session,
    messageId: string,
    accept: boolean,
  ) {
    if (!this.config.isAdmin(session.userId)) {
      return '你没有权限操作。';
    }
    this.logger.info(
      `${this.getUserDisplayName(session)} ${
        accept ? 'accepted' : 'rejected'
      } the invitation of ${messageId}.`,
    );
    try {
      await session.bot.handleGuildRequest(messageId, accept);
    } catch (e) {
      this.logger.error(
        `handleGuildRequest for ${messageId} failed: ${e.toString()}`,
      );
      return `操作失败，请查看服务器日志。`;
    }
    return `操作成功。`;
  }

  @OnPrivate()
  @UseCommand('verify.accept <messageId:string>', '同意邀请入群请求')
  @CommandDescription({ en: 'Accept guild invitation request' })
  @CommandUsage('需要具有处理权限。')
  accept(@PutSession() session: Session, @PutArg(0) messageId: string) {
    return this.handleRequest(session, messageId, true);
  }

  @OnPrivate()
  @UseCommand('verify.reject <messageId:string>', '拒绝邀请入群请求')
  @CommandDescription({ en: 'Reject guild invitation request' })
  @CommandUsage('需要具有处理权限。')
  reject(@PutSession() session: Session, @PutArg(0) messageId: string) {
    return this.handleRequest(session, messageId, false);
  }
}
