import yaml from "yaml";
import { promises as fs, promises } from "fs";
import util from "util"
import child_process from "child_process";
import os from "os";
import _, { has, last } from "underscore";
import { loadCsv, fileExists } from "./lib/utility";
import bunyan from "bunyan";

interface Config {
	csvPath: string;
	outputPath: string;
	pythonInterpreter: string;
	encoding: string;
}

interface NotebookConfig {
	password: string;
	base_url: string;
}

interface NotebookConfigFile {
	NotebookApp: NotebookConfig;
}

interface Instance {
	username: string;
	name: string;
	password: string;
}

interface InstanceData extends Instance {
	notebookConfig: NotebookConfigFile;
}

interface OutputData {
	instances: InstanceData[];
}

async function loadConfig() {
	const configPath = process.argv[2];
	const configRawData = await fs.readFile(configPath, "utf-8");
	const config: Config = yaml.parse(configRawData);
	return config;
}

class JupyterGenerator {
	config: Config;
	instances: Instance[];
	lastOutput: OutputData;
	log: bunyan;
	constructor(config: Config) {
		this.config = config;
		this.lastOutput = null;
		this.log = bunyan.createLogger({ name: "JupyterGenerator", level: "debug" });
	}
	async loadInstances() {
		this.instances = await loadCsv(this.config.csvPath, true, this.config.encoding || "UTF-8");
	}
	async readLastData() {
		this.log.debug(`Will read last data from ${this.config.outputPath}.`);
		try {
			const lastRawData = await fs.readFile(this.config.outputPath, "utf-8");
			this.lastOutput = yaml.parse(lastRawData);
		} catch (e) {
			this.log.warn(`Reading last data ${this.config.outputPath} failed: ${e.toString()}.`);
			return false;
		}
		return true;
	}
	async getHashedPassword(instance: Instance) {
		if (this.lastOutput) {
			const lastInstances = this.lastOutput.instances;
			const matchInstance = lastInstances.find(i => i.name === instance.name && i.password === instance.password);
			if (matchInstance) {
				return matchInstance.notebookConfig.NotebookApp.password;
			}
		}
		this.log.debug(`Generating password for instance ${instance.name}.`);
		const res = await util.promisify(child_process.exec)(`${this.config.pythonInterpreter || "python3"} -c 'print(__import__("notebook.auth", fromlist=[""]).passwd("${instance.password}"))'`);
		return res.stdout.split("\n")[0].trim();
	}
	async getDataFromInstance(instance: Instance): Promise<InstanceData> {
		const { username, name, password } = instance;
		const hashedPassword = await this.getHashedPassword(instance);
		return {
			username, name, password, notebookConfig: {
				NotebookApp: {
					password: hashedPassword,
					base_url: `/${name}/`
				}
			}
		}
	}
	async run() {
		await this.loadInstances();
		await this.readLastData();
		const instanceDatas = await Promise.all(this.instances.map(i => this.getDataFromInstance(i)));
		const outputData: OutputData = { instances: instanceDatas };
		//if (!await fileExists(this.config.outputPath)) {
		//	await fs.mkdir(this.config.outputPath, { recursive: true });
		//}
		await fs.writeFile(this.config.outputPath, yaml.stringify(outputData));
	}
}

async function main() {
	const config = await loadConfig();
	const jupyterGenerator = new JupyterGenerator(config);
	await jupyterGenerator.run();
}
main();
