import { TimeBase } from './TimeBase.entity';
import { Column, Entity, Index, PrimaryGeneratedColumn } from 'typeorm';
import path from 'path';
import hasha from 'hasha';
import { ReturnMessageDto } from '../dto/ReturnMessage.dto';

@Entity()
@Index((f) => [f.hash, f.name], { unique: true })
export class File extends TimeBase {
  @PrimaryGeneratedColumn({ type: 'int8' })
  id: number;

  @Index()
  @Column('varchar', { length: 128 })
  hash: string;

  @Index()
  @Column('varchar', { length: 128 })
  name: string;

  @Index()
  @Column('int8')
  branchId: number;

  get branchIdValue() {
    return parseInt(this.branchId.toString());
  }

  @Index()
  @Column('int4')
  size: number;

  url: string;

  //multerFile: Express.Multer.File;
  savePath: string;

  resolveFunction: ((_this: File) => void)[] = [];
  rejectFunction: ((error: any) => void)[] = [];

  resolve(user: string, repo: string) {
    this.url = this.getJsdelivrUrl(user, repo);
    this.savePath = undefined;
    for (const func of this.resolveFunction) {
      func(this);
    }
  }

  reject(error: any) {
    for (const func of this.rejectFunction) {
      func(error);
    }
  }

  async fromMulterFile(multerFile: Express.Multer.File) {
    this.savePath = multerFile.path;
    this.size = multerFile.size;
    const dotSplit = multerFile.originalname.split('/').pop().split('.');
    this.name = multerFile.originalname;
    if (this.name.length > 128) {
      throw new ReturnMessageDto(400, 'name too long').toException();
    }
    this.hash = await hasha.fromFile(multerFile.path);
  }

  get filename() {
    return `${this.hash}-${this.name}`;
  }

  getJsdelivrUrl(user: string, repo: string) {
    return `https://cdn.jsdelivr.net/gh/${user}/${repo}@${this.branch}/${this.filename}`;
  }

  get branch() {
    return this.branchIdValue.toString(36);
  }

  set branch(branchName: string) {
    this.branchId = parseInt(branchName, 36);
  }

  setSuffix(filename: string) {
    this.name = path.extname(filename);
    return this.name;
  }
}
