import { Connection, Repository } from 'typeorm';
import { InjectConnection } from '@nestjs/typeorm';
import { ConsoleLogger, Injectable } from '@nestjs/common';
import simpleGit, { ResetMode, SimpleGit } from 'simple-git';
import { ConfigService } from '@nestjs/config';
import path from 'path';
import { File } from './entities/File.entity';
import { promises as fs } from 'fs';
import * as os from 'os';
import delay from 'delay';
import { ReturnMessageDto } from './dto/ReturnMessage.dto';
import PQueue from 'p-queue';

@Injectable()
export class AppService extends ConsoleLogger {
  dbRepo: Repository<File>;
  git: SimpleGit;
  githubRepo: string;
  githubUser: string;
  githubToken: string;
  repoPath: string;
  maxBranchSize: number;
  commitQueue: File[] = [];
  inQueueQueue = new PQueue({ concurrency: 1 });
  queueHashmap = new Map<string, File>();

  repoUrl() {
    return `https://${this.githubToken}@github.com/${this.githubUser}/${this.githubRepo}.git`;
  }

  constructor(
    @InjectConnection('app')
    private db: Connection,
    private config: ConfigService,
  ) {
    super('app');
    this.githubRepo = config.get('GITHUB_REPO');
    this.githubUser = config.get('GITHUB_USER');
    this.githubToken = config.get('GITHUB_TOKEN');
    this.repoPath = config.get('REPO_PATH') || path.join(os.tmpdir(), 'repo');
    this.dbRepo = this.db.getRepository(File);
    this.maxBranchSize =
      parseInt(config.get('BRANCH_SIZE')) || 50 * 1024 * 1024;
    this.main().then();
  }

  async main() {
    /*try {
      await fs.mkdir(path.join(os.tmpdir(), 'uploads'), { recursive: true });
    } catch (e) {}*/
    try {
      await this.initRepo();
    } catch (e) {
      this.error(`Initialize failed: ${e.toString()}`);
    }
    while (true) {
      try {
        await this.mainLoop();
      } catch (e) {
        this.error(`Loop failed: ${e.toString()}`);
      }
    }
  }

  createGitInstance() {
    this.git = simpleGit({
      baseDir: this.repoPath,
      binary: this.config.get('GIT_BINARY') || 'git',
    });
  }

  gatherFileTasks(possibleSize: number) {
    const files: File[] = [];
    let size = 0;
    while (this.commitQueue.length > 0 && size <= possibleSize) {
      const file = this.commitQueue.pop();
      size += file.size;
      if (size <= possibleSize) {
        files.push(file);
      } else {
        this.commitQueue.push(file);
        break;
      }
    }
    return files;
  }

  async mainLoop() {
    //this.log('Running loop.');
    if (!this.commitQueue.length) {
      return delay(3000);
    }
    this.log(`Processing files.`);
    let checkoutResult = await this.checkoutCorrectBranch();
    let files = this.gatherFileTasks(this.maxBranchSize - checkoutResult.size);
    if (!files.length) {
      checkoutResult = await this.checkoutCorrectBranch(true);
      files = this.gatherFileTasks(this.maxBranchSize - checkoutResult.size);
    }
    if (!files.length) {
      this.error(`Cannot fetch any files, exiting.`);
      return;
    }
    this.log(`Will process ${files.length} files.`);
    try {
      this.log(`Moving files.`);
      for (const file of files) {
        file.branchId = checkoutResult.branchId;
      }
      await Promise.all(
        files.map((file) =>
          fs.rename(file.savePath, path.join(this.repoPath, file.filename)),
        ),
      );
      this.log(`Committing files.`);
      await this.git
        .add(files.map((f) => f.filename))
        .commit('upload')
        .push('origin', checkoutResult.branchId.toString(36), ['-u', '-f']);
      this.log(`Saving file entries to database.`);
      await this.dbRepo.save(files);
      this.log(`Finished processing files.`);
      for (const file of files) {
        file.resolve(this.githubUser, this.githubRepo);
      }
    } catch (e) {
      this.error(`Errored processing files: ${e.toString()}`);
      const filesToDelete: string[] = [];
      for (const file of files) {
        filesToDelete.push(file.savePath);
        filesToDelete.push(path.join(this.repoPath, file.filename));
        file.reject(new ReturnMessageDto(500, 'upload failed').toException());
      }
      try {
        await Promise.all(filesToDelete.map((path) => fs.unlink(path)));
      } catch (e) {}
    } finally {
      for (const file of files) {
        this.queueHashmap.delete(file.filename);
      }
    }
  }

  async initRepo() {
    this.log(`Initializing repo at ${this.repoPath}`);
    try {
      await fs.access(path.join(this.repoPath, '.git'));
      this.createGitInstance();
      this.log(`Repo exists, skipping.`);
    } catch (e) {
      try {
        await fs.access(this.repoPath);
      } catch (e) {
        await fs.mkdir(this.repoPath, { recursive: true });
      }
      this.createGitInstance();
      await this.git
        .init()
        .addRemote('origin', this.repoUrl())
        .addConfig('core.autocrlf', 'false', false);
      this.log(`Initialized repo at ${this.repoPath}`);
      await this.checkoutCorrectBranch(false, true);
      this.log(`Initializing finished.`);
    }
  }

  async checkoutCorrectBranch(
    forceNew?: boolean,
    fetch?: boolean,
    newBranchOffset = 1,
  ): Promise<{ branchId: number; size: number }> {
    const res = await this.getLatestBranchId(forceNew, newBranchOffset);
    const branchName = res.branchId.toString(36);
    if (!res.size) {
      this.log(`Checking out to a new branch ${branchName}`);
      try {
        await this.git.checkout(['--orphan', branchName]);
      } catch (e) {
        this.error(
          `Failed checking out to new branch ${branchName}, would retry with another branch: ${e.toString()}`,
        );
        return this.checkoutCorrectBranch(true, fetch, newBranchOffset + 1);
      }
      try {
        await this.git.raw(['rm', '-rf', '.']);
      } catch (e) {}
      this.log(`Checked out to a new branch ${branchName}`);
    } else {
      this.log(`Checking out existing branch ${branchName}`);
      try {
        if (fetch) {
          await this.git.fetch('origin', branchName);
        }
        await this.git.checkout(branchName, ['-f']);
        if (fetch) {
          await this.git.reset(ResetMode.HARD, [`origin/${branchName}`]);
        }
        this.log(`Checked out existing branch ${branchName}`);
      } catch (e) {
        this.error(
          `Checking out to existing branch failed, will checkout to a new branch: ${e.toString()}`,
        );
        return this.checkoutCorrectBranch(true, fetch);
      }
    }
    return res;
  }

  async getLatestBranchId(
    forceNew?: boolean,
    newBranchOffset = 1,
  ): Promise<{ branchId: number; size: number }> {
    const [latestFile] = await this.dbRepo.find({
      select: ['branchId'],
      order: { id: 'DESC' },
      take: 1,
    });
    if (!latestFile) {
      return { branchId: newBranchOffset, size: 0 };
    }
    const branchId = parseInt(latestFile.branchIdValue.toString());
    if (forceNew) {
      return { branchId: branchId + newBranchOffset, size: 0 };
    }
    const { totalSizeRaw } = await this.dbRepo
      .createQueryBuilder('file')
      .select('sum(file.size)', 'totalSizeRaw')
      .where('file.branchId = :branchId', { branchId })
      .getRawOne();
    const size = parseInt(totalSizeRaw);
    if (size >= this.maxBranchSize) {
      this.log(`Will switch to branch ${branchId.toString(36)}`);
      return { branchId: branchId + newBranchOffset, size: 0 };
    } else {
      this.log(`Will remain on branch ${branchId.toString(36)}`);
      return { branchId, size };
    }
  }

  async addFileToQueue(file: File) {
    return new Promise<File>(async (resolve, reject) => {
      await this.inQueueQueue.add(() => {
        const inQueueFile = this.queueHashmap.get(file.filename);
        if (inQueueFile) {
          inQueueFile.resolveFunction.push(resolve);
          inQueueFile.rejectFunction.push(reject);
        } else {
          this.log(`Queued file ${file.filename}`);
          file.resolveFunction.push(resolve);
          file.rejectFunction.push(reject);
          this.commitQueue.unshift(file);
          this.queueHashmap.set(file.filename, file);
        }
      });
    });
  }

  async upload(mutFile: Express.Multer.File) {
    const file = new File();
    await file.fromMulterFile(mutFile);
    const existingFile = await this.dbRepo.findOne({
      where: { hash: file.hash, name: file.name },
      select: ['hash', 'name', 'branchId'],
    });
    if (existingFile) {
      return new ReturnMessageDto(
        201,
        'success',
        existingFile.getJsdelivrUrl(this.githubUser, this.githubRepo),
      );
    }
    const uploadedFile = await this.addFileToQueue(file);
    return new ReturnMessageDto(201, 'success', uploadedFile.url);
  }
}
