import { ApiProperty } from '@nestjs/swagger';
import { HttpException } from '@nestjs/common';
import { User } from '../entities/User.entity';
import { PageSettingsWise } from './PageSettings.dto';

export interface BlankReturnMessage {
  statusCode: number;
  message: string;
  success: boolean;
}

export interface ReturnMessage<T> extends BlankReturnMessage {
  data?: T;
}

export class BlankReturnMessageDto implements BlankReturnMessage {
  @ApiProperty({ description: '返回状态' })
  statusCode: number;
  @ApiProperty({ description: '返回信息' })
  message: string;
  @ApiProperty({ description: '是否成功' })
  success: boolean;
  constructor(statusCode: number, message?: string) {
    this.statusCode = statusCode;
    this.message = message || 'success';
    this.success = statusCode < 400;
  }

  toException() {
    return new HttpException(this, this.statusCode);
  }
}

export class BlankPaginatedReturnMessageDto
  extends BlankReturnMessageDto
  implements PageSettingsWise {
  @ApiProperty({ description: '总数' })
  total: number;
  @ApiProperty({ description: '总页数' })
  totalPages: number;
  @ApiProperty({ description: '当前页' })
  pageCount: number;
  @ApiProperty({ description: '每页数量' })
  recordsPerPage: number;
  constructor(
    statusCode: number,
    message: string,
    total: number,
    pageSettings: PageSettingsWise,
  ) {
    super(statusCode, message);
    this.total = total;
    this.pageCount = pageSettings.pageCount;
    this.recordsPerPage = pageSettings.recordsPerPage;
    this.totalPages = Math.ceil(total / pageSettings.recordsPerPage);
  }
}

export class ReturnMessageDto<T>
  extends BlankReturnMessageDto
  implements ReturnMessage<T> {
  @ApiProperty({ description: '返回内容' })
  data?: T;
  constructor(statusCode: number, message?: string, data?: T) {
    super(statusCode, message);
    this.data = data;
  }
}

export class PaginatedReturnMessageDto<T>
  extends BlankPaginatedReturnMessageDto
  implements PageSettingsWise {
  @ApiProperty({ description: '返回内容' })
  data: T[];
  constructor(
    statusCode: number,
    message: string,
    data: T[],
    total: number,
    pageSettings: PageSettingsWise,
  ) {
    super(statusCode, message, total, pageSettings);
    this.data = data;
  }
}

export class StringReturnMessageDto
  extends BlankReturnMessageDto
  implements ReturnMessage<string> {
  @ApiProperty({ description: '返回内容' })
  data?: string;
}
