import {
  ApiBody,
  ApiCreatedResponse,
  ApiNoContentResponse,
  ApiOkResponse,
  ApiOperation,
  ApiParam,
  ApiProperty,
  OmitType,
  PartialType,
} from '@nestjs/swagger';
import { Body, Delete, Get, Patch, Post, Query, Type } from '@nestjs/common';
import {
  BlankPaginatedReturnMessageDto,
  BlankReturnMessageDto,
  PaginatedReturnMessageDto,
  ReturnMessageDto,
} from '../dto/ReturnMessage.dto';
import { TimeBase, TimeBaseFields } from '../entities/bases/TimeBase.entity';
import { ClassGetPipe, CreatePipe, UpdatePipe } from '../utility/pipes';

export function MergeMethodDecorators(
  decorators: MethodDecorator[],
): MethodDecorator {
  return (target: any, key: string, descriptor: PropertyDescriptor) => {
    decorators.forEach((decorator) => {
      decorator(target, key, descriptor);
    });
  };
}

export class CrudFactory<T extends TimeBase> {
  readonly createDto: Type<Omit<T, keyof T>>;
  readonly updateDto: Type<Partial<Omit<T, keyof T>>>;
  readonly entityReturnMessageDto: Type<ReturnMessageDto<T>>;
  readonly entityArrayReturnMessageDto: Type<PaginatedReturnMessageDto<T>>;

  constructor(
    public readonly entityClass: Type<T>,
    fieldsToOmit: (keyof T)[] = [],
    // eslint-disable-next-line @typescript-eslint/ban-types
    public readonly idType: Function = Number,
  ) {
    this.createDto = OmitType(this.entityClass, [
      ...TimeBaseFields,
      ...fieldsToOmit,
    ]);
    this.updateDto = PartialType(this.createDto);
    this.entityReturnMessageDto = class EntityReturnMessageDto extends (
      BlankReturnMessageDto
    ) {
      data: T;
    };
    ApiProperty({ type: this.entityClass })(
      this.entityReturnMessageDto.prototype,
      'data',
    );
    this.entityArrayReturnMessageDto = class EntityArrayReturnMessageDto extends (
      BlankPaginatedReturnMessageDto
    ) {
      data: T[];
    };
    ApiProperty({ type: [this.entityClass] })(
      this.entityArrayReturnMessageDto.prototype,
      'data',
    );
  }

  create(): MethodDecorator {
    return MergeMethodDecorators([
      Post(),
      ApiOperation({ summary: `Create a new ${this.entityClass.name}` }),
      ApiBody({ type: this.createDto }),
      ApiCreatedResponse({ type: this.entityReturnMessageDto }),
    ]);
  }

  createParam() {
    return Body(CreatePipe);
  }

  findOne(): MethodDecorator {
    return MergeMethodDecorators([
      Get(':id'),
      ApiOperation({ summary: `Find a ${this.entityClass.name} by id` }),
      ApiParam({ name: 'id', type: this.idType, required: true }),
      ApiOkResponse({ type: this.entityReturnMessageDto }),
    ]);
  }

  findAll(): MethodDecorator {
    return MergeMethodDecorators([
      Get(),
      ApiOperation({ summary: `Find all ${this.entityClass.name}` }),
      ApiOkResponse({ type: this.entityArrayReturnMessageDto }),
    ]);
  }

  findAllParam() {
    return Query(new ClassGetPipe(this.entityClass));
  }

  update(): MethodDecorator {
    return MergeMethodDecorators([
      Patch(':id'),
      ApiOperation({ summary: `Update a ${this.entityClass.name} by id` }),
      ApiParam({ name: 'id', type: this.idType, required: true }),
      ApiBody({ type: this.updateDto }),
      ApiOkResponse({ type: BlankReturnMessageDto }),
    ]);
  }

  updateParam() {
    return Body(UpdatePipe);
  }

  delete(): MethodDecorator {
    return MergeMethodDecorators([
      Delete(':id'),
      ApiOperation({ summary: `Delete a ${this.entityClass.name} by id` }),
      ApiParam({ name: 'id', type: this.idType, required: true }),
      ApiNoContentResponse({ type: BlankReturnMessageDto }),
    ]);
  }
}
