import { ConsoleLogger } from '@nestjs/common';
import { ClassConstructor } from 'class-transformer';
import {
  DeepPartial,
  DeleteResult,
  FindConditions,
  In,
  Repository,
  SelectQueryBuilder,
  UpdateResult,
} from 'typeorm';
import {
  BlankReturnMessageDto,
  PaginatedReturnMessageDto,
  ReturnMessageDto,
} from '../dto/ReturnMessage.dto';
import { QueryWise } from '../entities/interfaces/QueryWise';
import { camelCase } from 'typeorm/util/StringUtils';
import { DeletionWise, ImportWise } from '../entities/bases/TimeBase.entity';
import { PageSettingsFactory } from '../dto/PageSettings.dto';
import { ImportEntry } from 'src/dto/import-entry.dto';
import _ from 'lodash';

export type EntityId<T extends { id: any }> = T['id'];
export interface RelationDef {
  name: string;
  inner?: boolean;
}

export const Inner = (name: string): RelationDef => {
  return { name, inner: true };
};

export class CrudBase<
  T extends Record<string, any> & {
    id: string | number;
  } & QueryWise<T> &
    DeletionWise &
    ImportWise &
    PageSettingsFactory,
> extends ConsoleLogger {
  protected readonly entityName: string;
  constructor(
    protected entityClass: ClassConstructor<T>,
    protected repo: Repository<T>,
    protected entityRelations: (string | RelationDef)[] = [],
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    protected extraGetQuery: (qb: SelectQueryBuilder<T>) => void = (qb) => {},
  ) {
    super(`${entityClass.name} Service`);
    this.entityName = entityClass.name;
  }

  async batchCreate(
    ents: T[],
    beforeCreate?: (repo: Repository<T>) => Promise<void>,
    skipErrors = false,
  ) {
    const entsWithId = ents.filter((ent) => ent.id != null);
    const result = await this.repo.manager.transaction(async (mdb) => {
      let skipped: { result: string; entry: T }[] = [];
      const repo = mdb.getRepository(this.entityClass);

      if (entsWithId.length) {
        const existingEnts = await repo.find({
          where: { id: In(entsWithId.map((ent) => ent.id)) },
          select: ['id', 'deleteTime'],
          withDeleted: true,
        });
        if (existingEnts.length) {
          const existingEntsWithoutDeleteTime = existingEnts.filter(
            (ent) => ent.deleteTime == null,
          );
          const existingEntsWithDeleteTime = existingEnts.filter(
            (ent) => ent.deleteTime != null,
          );
          if (existingEntsWithoutDeleteTime.length) {
            if (!skipErrors) {
              throw new BlankReturnMessageDto(
                404,
                `${this.entityName} ID ${existingEntsWithoutDeleteTime.join(
                  ',',
                )} already exists`,
              ).toException();
            }
            const skippedEnts = ents.filter((ent) =>
              existingEntsWithoutDeleteTime.some((e) => e.id === ent.id),
            );
            skipped = skippedEnts.map((ent) => ({
              result: 'Already exists',
              entry: ent,
            }));
            const skippedEntsSet = new Set(skippedEnts);
            ents = ents.filter((ent) => !skippedEntsSet.has(ent));
          }
          if (existingEntsWithDeleteTime.length) {
            await repo.delete(
              existingEntsWithDeleteTime.map((ent) => ent.id) as any[],
            );
          }
        }
      }
      if (beforeCreate) {
        await beforeCreate(repo);
      }
      try {
        const results = await repo.save(ents as DeepPartial<T>[]);
        return {
          results,
          skipped,
        };
      } catch (e) {
        this.error(
          `Failed to create entity ${JSON.stringify(ents)}: ${e.toString()}`,
        );
        throw new BlankReturnMessageDto(500, 'internal error').toException();
      }
    });
    return new ReturnMessageDto(201, 'success', result);
  }

  async create(ent: T, beforeCreate?: (repo: Repository<T>) => Promise<void>) {
    const savedEnt = await this.repo.manager.transaction(async (mdb) => {
      const repo = mdb.getRepository(this.entityClass);
      if (ent.id != null) {
        const existingEnt = await repo.findOne({
          where: { id: ent.id },
          select: ['id', 'deleteTime'],
          withDeleted: true,
        });
        if (existingEnt) {
          if (existingEnt.deleteTime) {
            await repo.delete(existingEnt.id);
          } else {
            throw new BlankReturnMessageDto(
              404,
              `${this.entityName} ID ${ent.id} already exists`,
            ).toException();
          }
        }
      }
      if (beforeCreate) {
        await beforeCreate(repo);
      }
      try {
        return await repo.save(ent as DeepPartial<T>);
      } catch (e) {
        this.error(
          `Failed to create entity ${JSON.stringify(ent)}: ${e.toString()}`,
        );
        throw new BlankReturnMessageDto(500, 'internal error').toException();
      }
    });
    return new ReturnMessageDto(201, 'success', savedEnt);
  }

  protected get entityAliasName() {
    return camelCase(this.entityName);
  }

  protected applyRelationToQuery(
    qb: SelectQueryBuilder<T>,
    relation: RelationDef,
  ) {
    const { name } = relation;
    const relationUnit = name.split('.');
    const base =
      relationUnit.length === 1
        ? this.entityAliasName
        : relationUnit.slice(0, relationUnit.length - 1).join('_');
    const property = relationUnit[relationUnit.length - 1];
    const properyAlias = relationUnit.join('_');
    const methodName = relation.inner
      ? 'innerJoinAndSelect'
      : ('leftJoinAndSelect' as const);
    qb[methodName](`${base}.${property}`, properyAlias);
  }

  protected applyRelationsToQuery(qb: SelectQueryBuilder<T>) {
    for (const relation of this.entityRelations) {
      if (typeof relation === 'string') {
        this.applyRelationToQuery(qb, { name: relation });
      } else {
        this.applyRelationToQuery(qb, relation);
      }
    }
  }

  protected queryBuilder() {
    return this.repo.createQueryBuilder(this.entityAliasName);
  }

  async findOne(
    id: EntityId<T>,
    extraQuery: (qb: SelectQueryBuilder<T>) => void = () => {},
  ) {
    const query = this.queryBuilder()
      .where(`${this.entityAliasName}.id = :id`, { id })
      .take(1);
    this.applyRelationsToQuery(query);
    this.extraGetQuery(query);
    extraQuery(query);
    query.take(1);
    let ent: T;
    try {
      ent = await query.getOne();
    } catch (e) {
      const [sql, params] = query.getQueryAndParameters();
      this.error(
        `Failed to read entity ID ${id} with SQL ${sql} param ${params.join(
          ',',
        )}: ${e.toString()}`,
      );
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
    if (!ent) {
      throw new BlankReturnMessageDto(
        404,
        `${this.entityName} ID ${id} not found.`,
      ).toException();
    }
    return new ReturnMessageDto(200, 'success', ent);
  }

  async findAll(
    ent?: Partial<T>,
    extraQuery: (qb: SelectQueryBuilder<T>) => void = () => {},
  ) {
    const query = this.queryBuilder();
    if (ent) {
      ent.applyQuery(query, this.entityAliasName);
    }
    this.applyRelationsToQuery(query);
    this.extraGetQuery(query);
    extraQuery(query);
    try {
      const [ents, count] = await query.getManyAndCount();
      return new PaginatedReturnMessageDto(
        200,
        'success',
        ents,
        count,
        ent.getActualPageSettings(),
      );
    } catch (e) {
      const [sql, params] = query.getQueryAndParameters();
      this.error(
        `Failed to read entity cond ${JSON.stringify(
          ent,
        )} with SQL ${sql} param ${params.join(',')}: ${e.toString()}`,
      );
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
  }

  async update(
    id: EntityId<T>,
    entPart: Partial<T>,
    cond: FindConditions<T> = {},
  ) {
    let result: UpdateResult;
    try {
      result = await this.repo.update(
        {
          id,
          ...cond,
        },
        entPart,
      );
    } catch (e) {
      this.error(
        `Failed to update entity ID ${id} to ${JSON.stringify(
          entPart,
        )}: ${e.toString()}`,
      );
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
    if (!result.affected) {
      throw new BlankReturnMessageDto(
        404,
        `${this.entityName} ID ${id} not found.`,
      ).toException();
    }
    return new BlankReturnMessageDto(200, 'success');
  }

  async remove(
    id: EntityId<T>,
    hardDelete = false,
    cond: FindConditions<T> = {},
  ) {
    let result: UpdateResult | DeleteResult;
    const searchCond = {
      id,
      ...cond,
    };
    try {
      result = await (hardDelete
        ? this.repo.delete(searchCond)
        : this.repo.softDelete(searchCond));
    } catch (e) {
      this.error(`Failed to delete entity ID ${id}: ${e.toString()}`);
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
    if (!result.affected) {
      throw new BlankReturnMessageDto(
        404,
        `${this.entityName} ID ${id} not found.`,
      ).toException();
    }
    return new BlankReturnMessageDto(204, 'success');
  }

  async importEntities(
    ents: T[],
    extraChecking?: (ent: T) => string | Promise<string>,
  ): Promise<ReturnMessageDto<ImportEntry<T>[]>> {
    const invalidResults = _.compact(
      await Promise.all(
        ents.map(async (ent) => {
          const reason = ent.isValidInCreation();
          if (reason) {
            return { entry: ent, result: reason };
          }
          if (extraChecking) {
            const reason = await extraChecking(ent);
            if (reason) {
              return { entry: ent, result: reason };
            }
          }
        }),
      ),
    );
    const remainingEnts = ents.filter(
      (ent) => !invalidResults.find((result) => result.entry === ent),
    );
    await Promise.all(remainingEnts.map((ent) => ent.prepareForSaving()));
    const { data } = await this.batchCreate(remainingEnts, undefined, true);
    data.results.forEach((e) => {
      if (e.afterSaving) {
        e.afterSaving();
      }
    });
    const results = [
      ...invalidResults,
      ...data.skipped,
      ...data.results.map((e) => ({ entry: e, result: 'OK' })),
    ];
    return new ReturnMessageDto(201, 'success', results);
  }

  async exists(id: EntityId<T>): Promise<boolean> {
    const ent = await this.repo.findOne({ where: { id }, select: ['id'] });
    return !!ent;
  }
}
