import { Column, Index } from 'typeorm';
import { ApiProperty, ApiPropertyOptions } from '@nestjs/swagger';
import { ColumnWithLengthOptions } from 'typeorm/decorator/options/ColumnWithLengthOptions';
import { ColumnCommonOptions } from 'typeorm/decorator/options/ColumnCommonOptions';
import { ColumnEnumOptions } from 'typeorm/decorator/options/ColumnEnumOptions';
import {
  IsEnum,
  IsNotEmpty,
  IsOptional,
  IsString,
  ValidateIf,
} from 'class-validator';

export function MergePropertyDecorators(
  decs: PropertyDecorator[],
): PropertyDecorator {
  return (obj, key) => {
    for (const dec of decs) {
      dec(obj, key);
    }
  };
}

export const OptionalValidate = (...conitions: PropertyDecorator[]) =>
  MergePropertyDecorators([IsOptional(), ...conitions]);

export const StringColumn = (
  length = 32,
  description = 'unknown',
  defaultValue?: string,
  required = false,
  columnExtras: ColumnCommonOptions & ColumnWithLengthOptions = {},
  propertyExtras: ApiPropertyOptions = {},
) =>
  MergePropertyDecorators([
    Column('varchar', {
      length,
      default: defaultValue,
      nullable: !required && !defaultValue,
      ...columnExtras,
    }),
    ApiProperty({
      type: String,
      description,
      default: defaultValue,
      required: required && !defaultValue,
      ...propertyExtras,
    }),
    ...(required ? [] : [IsOptional()]),
    IsString(),
    IsNotEmpty(),
  ]);

export const EnumColumn = <T>(
  targetEnum: Record<string, T>,
  description = 'unknown',
  defaultValue?: T,
  required = false,
  columnExtras: ColumnCommonOptions & ColumnEnumOptions = {},
  swaggerExtras: ApiPropertyOptions = {},
) =>
  MergePropertyDecorators([
    Index(),
    Column('enum', {
      enum: targetEnum,
      default: defaultValue,
      nullable: !required && !defaultValue,
      ...columnExtras,
    }),
    ApiProperty({
      description,
      enum: targetEnum,
      default: defaultValue,
      required,
      ...swaggerExtras,
    }),
    ...(required ? [] : [IsOptional()]),
    IsEnum(targetEnum),
  ]);
