import { ConsoleLogger } from '@nestjs/common';
import { ClassConstructor } from 'class-transformer';
import {
  DeleteResult,
  FindConditions,
  In,
  Repository,
  SelectQueryBuilder,
  UpdateResult,
} from 'typeorm';
import {
  BlankReturnMessageDto,
  ReturnMessageDto,
} from '../dto/ReturnMessage.dto';
import { QueryWise } from '../entities/interfaces/QueryWise';
import { camelCase } from 'typeorm/util/StringUtils';
import { DeletionWise } from '../entities/bases/TimeBase.entity';

export type EntityId<T extends { id: any }> = T['id'];

export class CrudBase<
  T extends Record<string, any> & {
    id: string | number;
  } & QueryWise<T> &
    DeletionWise
> extends ConsoleLogger {
  protected readonly entityName: string;
  constructor(
    protected entityClass: ClassConstructor<T>,
    protected repo: Repository<T>,
    protected entityRelations: string[] = [],
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    protected extraGetQuery: (qb: SelectQueryBuilder<T>) => void = (qb) => {},
  ) {
    super(`${entityClass.name} Service`);
    this.entityName = entityClass.name;
  }

  async batchCreate(ents: T[], beforeCreate?: (repo: Repository<T>) => void) {
    const entsWithId = ents.filter((ent) => ent.id != null);
    const savedEnt = await this.repo.manager.transaction(async (mdb) => {
      const repo = mdb.getRepository(this.entityClass);

      if (entsWithId.length) {
        const existingEnts = await repo.find({
          where: { id: In(entsWithId.map((ent) => ent.id)) },
          select: ['id', 'deleteTime'],
          withDeleted: true,
        });
        if (existingEnts.length) {
          const existingEntsWithoutDeleteTime = existingEnts.filter(
            (ent) => ent.deleteTime == null,
          );
          if (existingEntsWithoutDeleteTime.length) {
            throw new BlankReturnMessageDto(
              404,
              `${this.entityName} ID ${existingEntsWithoutDeleteTime.join(
                ',',
              )} already exists`,
            ).toException();
          }
          await repo.delete(existingEnts.map((ent) => ent.id) as any[]);
        }
      }
      if (beforeCreate) {
        await beforeCreate(repo);
      }
      try {
        return await repo.save(ents);
      } catch (e) {
        this.error(
          `Failed to create entity ${JSON.stringify(ents)}: ${e.toString()}`,
        );
        throw new BlankReturnMessageDto(500, 'internal error').toException();
      }
    });
    return new ReturnMessageDto(201, 'success', savedEnt);
  }

  async create(ent: T, beforeCreate?: (repo: Repository<T>) => void) {
    const savedEnt = await this.repo.manager.transaction(async (mdb) => {
      const repo = mdb.getRepository(this.entityClass);
      if (ent.id != null) {
        const existingEnt = await repo.findOne({
          where: { id: ent.id },
          select: ['id', 'deleteTime'],
          withDeleted: true,
        });
        if (existingEnt) {
          if (existingEnt.deleteTime) {
            await repo.delete(existingEnt.id);
          } else {
            throw new BlankReturnMessageDto(
              404,
              `${this.entityName} ID ${ent.id} already exists`,
            ).toException();
          }
        }
      }
      if (beforeCreate) {
        await beforeCreate(repo);
      }
      try {
        return await repo.save(ent);
      } catch (e) {
        this.error(
          `Failed to create entity ${JSON.stringify(ent)}: ${e.toString()}`,
        );
        throw new BlankReturnMessageDto(500, 'internal error').toException();
      }
    });
    return new ReturnMessageDto(201, 'success', savedEnt);
  }

  protected get entityAliasName() {
    return camelCase(this.entityName);
  }

  protected applyRelationToQuery(qb: SelectQueryBuilder<T>, relation: string) {
    const relationUnit = relation.split('.');
    const base =
      relationUnit.length === 1
        ? this.entityAliasName
        : relationUnit.slice(0, relationUnit.length - 1).join('_');
    const property = relationUnit[relationUnit.length - 1];
    const properyAlias = relationUnit.join('_');
    qb.leftJoinAndSelect(`${base}.${property}`, properyAlias);
  }

  protected applyRelationsToQuery(qb: SelectQueryBuilder<T>) {
    for (const relation of this.entityRelations) {
      this.applyRelationToQuery(qb, relation);
    }
  }

  protected queryBuilder() {
    return this.repo.createQueryBuilder(this.entityAliasName);
  }

  async findOne(
    id: EntityId<T>,
    extraQuery: (qb: SelectQueryBuilder<T>) => void = () => {},
  ) {
    const query = this.queryBuilder()
      .where(`${this.entityAliasName}.id = :id`, { id })
      .take(1);
    this.applyRelationsToQuery(query);
    this.extraGetQuery(query);
    extraQuery(query);
    query.take(1);
    let ent: T;
    try {
      ent = await query.getOne();
    } catch (e) {
      const [sql, params] = query.getQueryAndParameters();
      this.error(
        `Failed to read entity ID ${id} with SQL ${sql} param ${params.join(
          ',',
        )}: ${e.toString()}`,
      );
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
    if (!ent) {
      throw new BlankReturnMessageDto(
        404,
        `${this.entityName} ID ${id} not found.`,
      ).toException();
    }
    return new ReturnMessageDto(200, 'success', ent);
  }

  async findAll(
    ent?: Partial<T>,
    extraQuery: (qb: SelectQueryBuilder<T>) => void = () => {},
  ) {
    const query = this.queryBuilder();
    if (ent) {
      ent.applyQuery(query, this.entityAliasName);
    }
    this.applyRelationsToQuery(query);
    this.extraGetQuery(query);
    extraQuery(query);
    try {
      return new ReturnMessageDto(200, 'success', await query.getMany());
    } catch (e) {
      const [sql, params] = query.getQueryAndParameters();
      this.error(
        `Failed to read entity cond ${JSON.stringify(
          ent,
        )} with SQL ${sql} param ${params.join(',')}: ${e.toString()}`,
      );
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
  }

  async update(
    id: EntityId<T>,
    entPart: Partial<T>,
    cond: FindConditions<T> = {},
  ) {
    let result: UpdateResult;
    try {
      result = await this.repo.update(
        {
          id,
          ...cond,
        },
        entPart,
      );
    } catch (e) {
      this.error(
        `Failed to update entity ID ${id} to ${JSON.stringify(
          entPart,
        )}: ${e.toString()}`,
      );
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
    if (!result.affected) {
      throw new BlankReturnMessageDto(
        404,
        `${this.entityName} ID ${id} not found.`,
      ).toException();
    }
    return new BlankReturnMessageDto(200, 'success');
  }

  async remove(
    id: EntityId<T>,
    hardDelete = false,
    cond: FindConditions<T> = {},
  ) {
    let result: UpdateResult | DeleteResult;
    const searchCond = {
      id,
      ...cond,
    };
    try {
      result = await (hardDelete
        ? this.repo.delete(searchCond)
        : this.repo.softDelete(searchCond));
    } catch (e) {
      this.error(`Failed to delete entity ID ${id}: ${e.toString()}`);
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
    if (!result.affected) {
      throw new BlankReturnMessageDto(
        404,
        `${this.entityName} ID ${id} not found.`,
      ).toException();
    }
    return new BlankReturnMessageDto(204, 'success');
  }
}
