import { Column, Index } from 'typeorm';
import { ApiProperty, ApiPropertyOptions } from '@nestjs/swagger';
import { ColumnWithLengthOptions } from 'typeorm/decorator/options/ColumnWithLengthOptions';
import { ColumnCommonOptions } from 'typeorm/decorator/options/ColumnCommonOptions';
import { ColumnEnumOptions } from 'typeorm/decorator/options/ColumnEnumOptions';
import {
  IsEnum,
  IsInt,
  IsNotEmpty,
  IsOptional,
  IsString,
  Min,
} from 'class-validator';
import { ColumnWithWidthOptions } from 'typeorm/decorator/options/ColumnWithWidthOptions';
import { BigintTransformer } from '../utility/bigint-transform';

export function MergePropertyDecorators(
  decs: PropertyDecorator[],
): PropertyDecorator {
  return (obj, key) => {
    for (const dec of decs) {
      dec(obj, key);
    }
  };
}

export const OptionalValidate = (...conitions: PropertyDecorator[]) =>
  MergePropertyDecorators([IsOptional(), ...conitions]);

export const StringColumn = (
  length = 32,
  description = 'unknown',
  defaultValue?: string,
  required = false,
  columnExtras: ColumnCommonOptions & ColumnWithLengthOptions = {},
  propertyExtras: ApiPropertyOptions = {},
) =>
  MergePropertyDecorators([
    Column('varchar', {
      length,
      default: defaultValue,
      nullable: !required && defaultValue == null,
      ...columnExtras,
    }),
    ApiProperty({
      type: String,
      description,
      default: defaultValue,
      required: required && defaultValue == null,
      ...propertyExtras,
    }),
    ...(required ? [] : [IsOptional()]),
    IsString(),
    IsNotEmpty(),
  ]);

export const IntColumn = (
  type: 'int' | 'smallint' | 'bigint' | 'tinyint' = 'int',
  unsigned = false,
  description = 'unknown',
  defaultValue?: number,
  required = false,
  columnExtras: ColumnCommonOptions & ColumnWithWidthOptions = {},
  propertyExtras: ApiPropertyOptions = {},
) =>
  MergePropertyDecorators([
    Column(type, {
      default: defaultValue,
      nullable: !required && defaultValue == null,
      unsigned,
      ...(type === 'bigint' ? { transformer: new BigintTransformer() } : {}),
      ...columnExtras,
    }),
    ApiProperty({
      type: Number,
      description,
      default: defaultValue,
      required: required && defaultValue == null,
      ...propertyExtras,
    }),
    ...(required ? [] : [IsOptional()]),
    IsInt(),
    ...(unsigned ? [Min(0)] : []),
  ]);

export const EnumColumn = <T>(
  targetEnum: Record<string, T>,
  description = 'unknown',
  defaultValue?: T,
  required = false,
  columnExtras: ColumnCommonOptions & ColumnEnumOptions = {},
  swaggerExtras: ApiPropertyOptions = {},
) =>
  MergePropertyDecorators([
    Index(),
    Column('enum', {
      enum: targetEnum,
      default: defaultValue,
      nullable: !required && !defaultValue,
      ...columnExtras,
    }),
    ApiProperty({
      description,
      enum: targetEnum,
      default: defaultValue,
      required,
      ...swaggerExtras,
    }),
    ...(required ? [] : [IsOptional()]),
    IsEnum(targetEnum),
  ]);
