import { Injectable, ConsoleLogger } from '@nestjs/common';
import { InjectConnection } from '@nestjs/typeorm';
import { Connection } from 'typeorm';
import { User } from '../entities/User.entity';
import { Group } from '../entities/Group.entity';
import { App, AppConfig } from 'koishi';
import { AppService } from '../app.service';
import * as koishiCommonPlugin from 'koishi-plugin-common';
import * as adapter from 'koishi-adapter-onebot';
import { CQBot } from 'koishi-adapter-onebot';

const __ = typeof adapter; // just for import
@Injectable()
export class BotService extends ConsoleLogger {
  bot: App;
  botConfig: AppConfig;
  constructor(
    @InjectConnection('app')
    private db: Connection,
    private readonly appService: AppService,
  ) {
    super('bot');
    this.initializeBot();
  }
  async initializeBot() {
    this.botConfig = {
      type: 'onebot:ws',
      selfId: process.env.CQ_ID,
      server: process.env.CQ_SERVER,
      token: process.env.CQ_TOKEN,
      prefix: process.env.CQ_PREFIX || '.',
    };
    this.bot = new App(this.botConfig);
    this.bot.plugin(koishiCommonPlugin, {
      onFriendRequest: true,
    });
    this.loadBotRouters();
    await this.bot.start();
    this.log.log(`Bot started.`);
  }
  loadBotRouters() {
    // all middlewares should be here.
    this.bot.command('echo <msg:text>').action((argv, msg) => {
      return msg;
    });
  }

  async findOrCreateUser(id: string, name?: string) {
    const repo = this.db.getRepository(User);
    let ent = await repo.findOne({ where: { id } });
    if (ent) {
      if (!ent.name && name) {
        ent.name = name;
        return await repo.save(ent);
      } else {
        return ent;
      }
    }
    ent = new User();
    ent.id = id;
    ent.name = name;
    try {
      ent = await repo.save(ent);
    } catch (e) {
      this.log.error(`Failed to save user ${id}: ${e.toString()}`);
      return null;
    }
    return ent;
  }
  async findOrCreateGroup(id: string) {
    const repo = this.db.getRepository(Group);
    let ent = await repo.findOne({ where: { id } });
    if (ent) {
      return ent;
    }
    ent = new Group();
    ent.id = id;
    try {
      ent = await repo.save(ent);
    } catch (e) {
      this.log.error(`Failed to save group ${id}: ${e.toString()}`);
      return null;
    }
    return ent;
  }
  getBot() {
    return (this.bot.bots[0] as unknown) as CQBot;
  }

  async sendPrivateMessage(id: string, message: string) {
    try {
      await this.getBot().$sendPrivateMsg(id, message);
      return true;
    } catch (e) {
      this.log.error(
        `Failed to send private message ${message} to ${id}: ${e.toString()}`,
      );
      return false;
    }
  }

  async sendGroupMessage(id: string, message: string) {
    try {
      await this.getBot().$sendGroupMsg(id, message);
      return true;
    } catch (e) {
      this.log.error(
        `Failed to send group message ${message} to ${id}: ${e.toString()}`,
      );
      return false;
    }
  }
}
