import { ConsoleLogger } from '@nestjs/common';
import { ClassConstructor } from 'class-transformer';
import { Repository, SelectQueryBuilder, UpdateResult } from 'typeorm';
import {
  BlankReturnMessageDto,
  ReturnMessageDto,
} from '../dto/ReturnMessage.dto';
import { DeletionWise } from '../entities/bases/DeletionBase.entity';
import { QueryWise } from '../entities/interfaces/QueryWise';
import { camelCase } from 'typeorm/util/StringUtils';

export type EntityId<T> = T extends { id: string }
  ? string
  : T extends { id: number }
  ? number
  : never;

export class CrudBase<
  T extends Record<string, any> & {
    id: string | number;
  } & QueryWise<T> &
    DeletionWise
> extends ConsoleLogger {
  protected readonly entityName: string;
  constructor(
    protected entityClass: ClassConstructor<T>,
    protected repo: Repository<T>,
    protected entityRelations: string[] = [],
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    protected extraGetQuery: (qb: SelectQueryBuilder<T>) => void = (qb) => {},
  ) {
    super(`${entityClass.name} Service`);
    this.entityName = entityClass.name;
  }

  async create(ent: T) {
    const savedEnt = await this.repo.manager.transaction(async (mdb) => {
      const repo = mdb.getRepository(this.entityClass);
      if (ent.id != null) {
        const existingEnt = await repo.findOne({
          where: { id: ent.id },
          take: 1,
          select: ['id', 'isDeleted'],
        });
        if (existingEnt) {
          if (existingEnt.isDeleted) {
            await repo.delete({ id: ent.id });
          } else {
            throw new BlankReturnMessageDto(
              404,
              'already exists',
            ).toException();
          }
        }
      }
      try {
        return await repo.save(ent);
      } catch (e) {
        this.error(
          `Failed to create entity ${JSON.stringify(ent)}: ${e.toString()}`,
        );
        throw new BlankReturnMessageDto(500, 'internal error').toException();
      }
    });
    return new ReturnMessageDto(201, 'success', savedEnt);
  }

  protected get entityAliasName() {
    return `${camelCase(this.entityName)}_0`;
  }
  protected applyRelationToQuery(qb: SelectQueryBuilder<T>, relation: string) {
    const relationUnit = relation.split('.');
    const base =
      relationUnit.length === 1
        ? this.entityAliasName
        : `${relationUnit[relationUnit.length - 2]}_${relationUnit.length - 1}`;
    const property = relationUnit[relationUnit.length - 1];
    const properyAlias = `${property}_${relationUnit.length}`;
    qb.leftJoinAndSelect(
      `${base}.${property}`,
      properyAlias,
      `${properyAlias}.isDeleted = false`,
    );
  }

  protected applyRelationsToQuery(qb: SelectQueryBuilder<T>) {
    for (const relation of this.entityRelations) {
      this.applyRelationToQuery(qb, relation);
    }
  }

  private queryBuilder() {
    return this.repo.createQueryBuilder(this.entityAliasName);
  }

  async findOne(id: EntityId<T>) {
    const query = this.queryBuilder()
      .where(`${this.entityAliasName}.id = :id`, { id })
      .andWhere(`${this.entityAliasName}.isDeleted = false`)
      .take(1);
    this.applyRelationsToQuery(query);
    this.extraGetQuery(query);
    try {
      const ent = await query.getOne();
      if (!ent) {
        throw new BlankReturnMessageDto(
          404,
          `ID ${id} not found.`,
        ).toException();
      }
      return new ReturnMessageDto(200, 'success', ent);
    } catch (e) {
      const [sql, params] = query.getQueryAndParameters();
      this.error(
        `Failed to read entity ID ${id} with SQL ${sql} param ${params.join(
          ',',
        )}: ${e.toString()}`,
      );
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
  }

  async findAll(ent?: T) {
    const query = this.queryBuilder();
    if (ent) {
      ent.applyQuery(query, this.entityAliasName);
    }
    this.applyRelationsToQuery(query);
    this.extraGetQuery(query);
    try {
      return new ReturnMessageDto(200, 'success', await query.getMany());
    } catch (e) {
      const [sql, params] = query.getQueryAndParameters();
      this.error(
        `Failed to read entity cond ${JSON.stringify(
          ent,
        )} with SQL ${sql} param ${params.join(',')}: ${e.toString()}`,
      );
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
  }

  async update(id: EntityId<T>, entPart: Partial<T>) {
    let result: UpdateResult;
    try {
      result = await this.repo.update({ id, isDeleted: false }, entPart);
    } catch (e) {
      this.error(
        `Failed to create entity ID ${id} to ${JSON.stringify(
          entPart,
        )}: ${e.toString()}`,
      );
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
    if (!result.affected) {
      throw new BlankReturnMessageDto(404, `ID ${id} not found.`).toException();
    }
    return new BlankReturnMessageDto(200, 'success');
  }

  async remove(id: EntityId<T>) {
    let result: UpdateResult;
    try {
      result = await this.repo.update({ id, isDeleted: false }, {
        isDeleted: true,
      } as Partial<T>);
    } catch (e) {
      this.error(`Failed to delete entity ID ${id}: ${e.toString()}`);
      throw new BlankReturnMessageDto(500, 'internal error').toException();
    }
    if (!result.affected) {
      throw new BlankReturnMessageDto(404, `ID ${id} not found.`).toException();
    }
    return new BlankReturnMessageDto(200, 'success');
  }
}
