import { Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { HttpService } from '@nestjs/axios';
import { lastValueFrom } from 'rxjs';
import { BlankReturnMessageDto, ReturnMessage } from '../dto/ReturnMessage.dto';
import FormData from 'form-data';
import { PostUrlDto } from '../dto/PostUrl.dto';
import { MiddlewareInfoDto } from '../dto/MiddlewareInfo.dto';
import { UtilityService } from '../utility/utility/utility.service';
import { MiddlewareService } from '../abstract/service';

@Injectable()
export class JsdService extends MiddlewareService {
  private jsdUrl: string;
  private jsdIdentifier: string;
  constructor(
    config: ConfigService,
    http: HttpService,
    private utility: UtilityService,
  ) {
    super('jsd', http);
    this.jsdUrl = config.get('JSD_URL');
    this.jsdIdentifier = config.get('JSD_IDENTIFIER') || `jsd-${this.jsdUrl}`;
  }

  async info() {
    if (!this.jsdUrl) {
      throw new BlankReturnMessageDto(
        404,
        'JSDelivr is not configured.',
      ).toException();
    }
    const info = new MiddlewareInfoDto();
    info.identifier = this.jsdIdentifier;
    info.maxSize = 50 * 1024 * 1024;
    return info;
  }

  async upload(urlDto: PostUrlDto) {
    if (!this.jsdUrl) {
      throw new BlankReturnMessageDto(
        404,
        'JSDelivr is not configured.',
      ).toException();
    }
    const streamData = await this.utility.getStreamFromUrl(urlDto.url);
    //streamData.data.pause();
    const form = new FormData();
    form.append('file', streamData.data, {
      knownLength: urlDto.size || undefined,
      contentType:
        urlDto.customMime || streamData.headers['content-type'] || undefined,
      filename: urlDto.getFilename(),
    });
    try {
      //streamData.data.resume();
      const { data } = await lastValueFrom(
        this.http.post<ReturnMessage<string>>(`${this.jsdUrl}/upload`, form, {
          responseType: 'json',
          headers: form.getHeaders(),
          maxBodyLength: 4 * 1024 ** 3,
          maxContentLength: 4 * 1024 ** 3,
        }),
      );
      return data.data;
    } catch (e) {
      this.error(`Failed uploading ${urlDto.url}: ${e.toString()} ${e.data}`);
      throw new BlankReturnMessageDto(500, 'upload fail').toException();
    }
  }
}
