import { Injectable } from '@nestjs/common';
import { MiddlewareService } from '../abstract/service';
import { ConfigService } from '@nestjs/config';
import { HttpService } from '@nestjs/axios';
import { UtilityService } from '../utility/utility/utility.service';
import { BlankReturnMessageDto } from '../dto/ReturnMessage.dto';
import { MiddlewareInfoDto } from '../dto/MiddlewareInfo.dto';
import { lastValueFrom } from 'rxjs';
import moment, { Moment } from 'moment';
import { S3Client } from '@aws-sdk/client-s3';
import { Upload } from '@aws-sdk/lib-storage';
import { Aragami } from 'aragami';

interface OTTResponse<T> {
  code: number;
  message: string;
  data: T;
}

interface LoginResultData {
  expire: string;
  login_type: number;
  refresh_token_expire_time: number;
  token: string;
}

interface UploadCompleteReq {
  Info?: ServerFileInfo;
}

interface UploadReq extends UploadCompleteReq {
  AccessKeyId: string;
  SecretAccessKey: string;
  SessionToken: string;
  Expiration: string;
  Key: string;
  Bucket: string;
  FileId: number;
  Reuse: boolean;
  UploadId: string;
  DownloadUrl: string;
}

interface FileInfo {
  etag: string;
  fileId: number;
  fileName: string;
  s3keyFlag: string;
  size: number;
}

interface ServerFileInfo {
  FileId: number;
  FileName: string;
  Type: number;
  Size: number;
  ContentType: string;
  S3KeyFlag: string;
  CreateAt: string;
  UpdateAt: string;
  Hidden: boolean;
  Etag: string;
  Status: number;
  ParentFileId: number;
  Category: number;
  PunishFlag: number;
  ParentName: string;
  DownloadUrl: string;
}

function adaptFileInfo(info: ServerFileInfo): FileInfo {
  return {
    etag: info.Etag,
    fileId: info.FileId,
    fileName: info.FileName,
    s3keyFlag: info.S3KeyFlag,
    size: info.Size,
  };
}

@Injectable()
export class OneTwoThreeService extends MiddlewareService {
  username = this.config.get('OTT_USERNAME');
  password = this.config.get('OTT_PASSWORD');
  directory = parseInt(this.config.get('OTT_DIRECTORY'));
  identifier =
    this.config.get('OTT_IDENTIFIER') ||
    `ott-${this.username || 'undefined'}-${this.directory}`;
  constructor(
    private config: ConfigService,
    http: HttpService,
    private utility: UtilityService,
  ) {
    super('one-two-three', http);
  }

  token: string;
  tokenExpireTime: Moment;

  private aragami = new Aragami();

  private async loginProcess(relogin = false) {
    if (
      !relogin &&
      this.token &&
      this.tokenExpireTime &&
      this.tokenExpireTime.isAfter(moment())
    ) {
      return this.token;
    }
    const { data } = await lastValueFrom(
      this.http.post<OTTResponse<LoginResultData>>(
        'https://www.123pan.com/api/user/sign_in',
        { passport: this.username, password: this.password },
        { responseType: 'json' },
      ),
    );
    if (data.code < 300) {
      this.token = data.data.token;
      this.tokenExpireTime = moment.unix(data.data.refresh_token_expire_time);
      return data.data.token;
    }
    this.error(`Login failed: ${data.message}`);
    throw new BlankReturnMessageDto(data.code, data.message).toException();
  }

  private async login() {
    return this.aragami.lock('ott-login', () => this.loginProcess());
  }

  async info() {
    if (!this.username) {
      throw new BlankReturnMessageDto(
        404,
        'ott is not configured.',
      ).toException();
    }
    return new MiddlewareInfoDto({
      identifier: this.identifier,
      maxSize: 50 * 1024 ** 3,
      callback: true,
      // singleton: true,
    });
  }

  async upload(info) {
    try {
      const fileName = info.getFilename();
      this.log(`Uploading ${info.url} as ${fileName}`);
      const uploadReq = await lastValueFrom(
        this.http.post<OTTResponse<UploadReq>>(
          'https://www.123pan.com/api/file/upload_request',
          {
            /// https://github.com/alist-org/alist/blob/main/drivers/123/driver.go#L222
            driveId: 0,
            duplicate: 2,
            etag: info.hash?.slice(0, 32),
            parentFileId: this.directory,
            fileName: fileName,
            size: info.size,
            type: 0,
          },
          {
            responseType: 'json',
            headers: {
              Authorization: `Bearer ${await this.login()}`,
            },
          },
        ),
      );
      if (uploadReq.data.code > 300) {
        throw new Error(`Upload request failed: ${uploadReq.data.message}`);
      }
      if (!uploadReq.data.data.AccessKeyId) {
        // already uploaded
        // return JSON.stringify(adaptFileInfo(uploadReq.data.data.Info));
        const existing = JSON.stringify(
          adaptFileInfo(uploadReq.data.data.Info),
        );
        this.log(`File ${fileName} exists as ${existing}, skipping.`);
        return existing;
      }

      const s3 = new S3Client({
        credentials: {
          accessKeyId: uploadReq.data.data.AccessKeyId,
          secretAccessKey: uploadReq.data.data.SecretAccessKey,
          sessionToken: uploadReq.data.data.SessionToken,
        },
        region: '123pan',
        endpoint: 'https://file.123pan.com',
        forcePathStyle: true,
      });
      const streamInfo = await this.utility.getStreamFromUrl(info.url);
      const upload = new Upload({
        client: s3,
        params: {
          Bucket: uploadReq.data.data.Bucket,
          Key: uploadReq.data.data.Key,
          Body: streamInfo.data,
          ContentType: info.customMime || streamInfo.headers['content-type'],
          ContentLength: info.size,
        },
      });
      await upload.done();
      const completeReq = await lastValueFrom(
        this.http.post<OTTResponse<UploadCompleteReq>>(
          'https://www.123pan.com/api/file/upload_complete',
          { fileId: uploadReq.data.data.FileId },
          {
            responseType: 'json',
            headers: {
              Authorization: `Bearer ${await this.login()}`,
            },
          },
        ),
      );
      const uploadedUrl = JSON.stringify(
        adaptFileInfo(completeReq.data.data.Info),
      );
      this.log(`Uploaded ${fileName} as ${uploadedUrl}`);
      return uploadedUrl;
    } catch (e) {
      this.error(`Upload failed: ${e.message}`);
      throw new BlankReturnMessageDto(500, e.message).toException();
    }
  }

  async getDownloadUrl(fileInfo: FileInfo, ip: string) {
    const downloadReq = await lastValueFrom(
      this.http.post<OTTResponse<UploadReq>>(
        'https://www.123pan.com/api/file/download_info',
        { ...fileInfo, driveId: 0, type: 0 },
        {
          responseType: 'json',
          headers: {
            Authorization: `Bearer ${await this.login()}`,
            'X-Forwarded-For': ip,
          },
        },
      ),
    );
    if (downloadReq.data.code > 300) {
      throw new Error(`Download request failed: ${downloadReq.data.message}`);
    }
    return downloadReq.data.data.DownloadUrl;
  }

  async parse302(url: string, ip: string) {
    if (url.includes('?params=')) {
      const params = url.split('?params=')[1];
      const decoded = Buffer.from(params, 'base64').toString('utf-8');
      return this.parse302(decoded, ip);
    }
    const tmpLinkReq = await lastValueFrom(
      this.http.head(url, {
        maxRedirects: 0,
        validateStatus: (c) => c === 302 || c === 301 || c === 200,
        headers: {
          'X-Forwarded-For': ip,
        },
      }),
    );
    if (tmpLinkReq.status !== 200) {
      return tmpLinkReq.headers.location;
    }
    return url;
  }

  async download(info, ip: string) {
    try {
      const fileInfo = JSON.parse(info.url) as FileInfo;
      return await this.parse302(
        await this.getDownloadUrl(fileInfo, info.ip || ip),
        info.ip || ip,
      );
    } catch (e) {
      this.error(`Download failed: ${e.message}`);
      throw new BlankReturnMessageDto(500, e.message).toException();
    }
  }
}
