import { Injectable, ConsoleLogger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { HttpService } from '@nestjs/axios';
import internal from 'stream';
import { lastValueFrom } from 'rxjs';
import {
  BlankReturnMessageDto,
  ReturnMessage,
  ReturnMessageDto,
} from './dto/ReturnMessage.dto';
import FormData from 'form-data';
import { PostUrlDto } from './dto/PostUrl.dto';
import { MiddlewareInfoDto } from './dto/MiddlewareInfo.dto';

@Injectable()
export class AppService extends ConsoleLogger {
  private jsdUrl: string;
  private jsdIdentifier: string;
  constructor(config: ConfigService, private http: HttpService) {
    super('app');
    this.jsdUrl = config.get('JSD_URL');
    this.jsdIdentifier = config.get('JSD_IDENTIFIER') || `jsd-${this.jsdUrl}`;
  }

  private async getStreamFromUrl(url: string) {
    try {
      const ret = await lastValueFrom(
        await this.http.get<internal.Readable>(url, {
          responseType: 'stream',
          timeout: 30000,
        }),
      );
      return ret;
    } catch (e) {
      throw new BlankReturnMessageDto(404, 'URL not available').toException();
    }
  }

  jsdInfo() {
    if (!this.jsdUrl) {
      throw new BlankReturnMessageDto(
        404,
        'JSDelivr is not configured.',
      ).toException();
    }
    const info = new MiddlewareInfoDto();
    info.identifier = this.jsdIdentifier;
    info.maxSize = 50 * 1024 * 1024;
    return new ReturnMessageDto(200, 'success', info);
  }

  async jsd(urlDto: PostUrlDto) {
    if (!this.jsdUrl) {
      throw new BlankReturnMessageDto(
        404,
        'JSDelivr is not configured.',
      ).toException();
    }
    const streamData = await this.getStreamFromUrl(urlDto.url);
    //streamData.data.pause();
    const form = new FormData();
    form.append('file', streamData.data, {
      knownLength: urlDto.size || undefined,
      contentType:
        urlDto.customMime || streamData.headers['content-type'] || undefined,
      filename: urlDto.getFilename(),
    });
    try {
      //streamData.data.resume();
      const { data } = await lastValueFrom(
        this.http.post<ReturnMessage<string>>(`${this.jsdUrl}/upload`, form, {
          responseType: 'json',
          headers: form.getHeaders(),
          maxBodyLength: Infinity,
        }),
      );
      return new ReturnMessageDto(201, 'success', data.data);
    } catch (e) {
      this.error(`Failed uploading ${urlDto.url}: ${e.toString()} ${e.data}`);
      throw new BlankReturnMessageDto(500, 'upload fail').toException();
    }
  }
}
