import { ConsoleLogger, Injectable } from '@nestjs/common';
import { TalkDto } from './talk.dto';
import { ConversationService } from '../conversation/conversation.service';
import { v4 as uuid } from 'uuid';
import { ConversationDto } from '../conversation/conversation.dto';
import { BetterLock } from 'better-lock/dist/better_lock';
import { BlankReturnMessageDto } from '../dto/ReturnMessage.dto';
import { AccountPoolService } from '../account-pool/account-pool.service';
import { DavinciService } from '../davinci/davinci.service';

@Injectable()
export class ChatgptService extends ConsoleLogger {
  constructor(
    private conversationService: ConversationService,
    private accountPoolService: AccountPoolService,
    private davinciService: DavinciService,
  ) {
    super('ChatGPT');
  }

  async chatProcess(
    question: TalkDto,
    failedAccounts: string[] = [],
  ): Promise<ConversationDto> {
    const session = question.session || uuid();
    const previousConversation = failedAccounts.length
      ? undefined
      : await this.conversationService.getConversation(session);
    const account = this.accountPoolService.getAccount(
      previousConversation?.account,
      failedAccounts,
    );
    if (
      !account ||
      (failedAccounts.length > 1 && (await this.davinciService.isAvailable()))
    ) {
      /*throw new BlankReturnMessageDto(
        500,
        'No available accounts',
      ).toException();*/
      // fallback to davinci
      const davinciResponse = await this.davinciService.chat(question.text);
      const dto = new ConversationDto();
      dto.session = '';
      dto.isNewConversation = true;
      dto.text = davinciResponse.text.replace(/^<!--(.*)-->$/gm, '');
      return dto;
    }
    const isNewConversation =
      !previousConversation ||
      account.getEmail() !== previousConversation.account;
    this.log(`Processing chat for ${session} with ${account.getEmail()}`);
    const result = await account.sendMessage(question.text, {
      ...(isNewConversation
        ? {}
        : {
            conversationId: previousConversation.conversationId,
            parentMessageId: previousConversation.messageId,
          }),
      timeoutMs: 120000,
    });
    if (!result) {
      return this.chatProcess(question, [
        ...failedAccounts,
        account.getEmail(),
      ]);
    }
    await this.conversationService.saveConversation(
      session,
      result,
      account.getEmail(),
    );
    const dto = new ConversationDto();
    dto.session = session;
    dto.isNewConversation = isNewConversation;
    dto.text = result.response.replace(/^<!--(.*)-->$/gm, '');
    return dto;
  }

  private sessionLock = new BetterLock();

  async chat(dto: TalkDto) {
    return dto
      ? this.sessionLock.acquire(dto.session, () => this.chatProcess(dto))
      : this.chatProcess(dto);
  }

  async hasConversation(session: string) {
    const conversation = await this.conversationService.getConversation(
      session,
    );
    return (
      conversation && this.accountPoolService.hasAccount(conversation.account)
    );
  }
}
