import {
  Body,
  Controller,
  Get,
  Headers,
  Post,
  Query,
  ValidationPipe,
} from '@nestjs/common';
import { ChatgptService } from './chatgpt/chatgpt.service';
import {
  BlankReturnMessageDto,
  ReturnMessageDto,
} from './dto/ReturnMessage.dto';
import {
  ConversationDto,
  HasConversationDto,
} from './conversation/conversation.dto';
import {
  ApiBody,
  ApiCreatedResponse,
  ApiOkResponse,
  ApiOperation,
} from '@nestjs/swagger';
import { TalkDto } from './chatgpt/talk.dto';
import { SessionDto } from './conversation/session.dto';
import { ConversationService } from './conversation/conversation.service';
import { AuthService } from './auth/auth.service';
import { AccountPoolStatusDto } from './account-pool/account-pool-status.dto';
import { AccountPoolService } from './account-pool/account-pool.service';

const ResponseDto = ReturnMessageDto(ConversationDto);
const StatusDto = ReturnMessageDto(AccountPoolStatusDto);
const hasConversationResponse = ReturnMessageDto(HasConversationDto);

@Controller()
export class AppController {
  constructor(
    private readonly chatgpt: ChatgptService,
    private readonly conversation: ConversationService,
    private readonly authService: AuthService,
    private readonly accountPool: AccountPoolService,
  ) {}

  @Post('chat')
  @ApiOperation({ summary: 'Chat with ChatGPT' })
  @ApiBody({ type: TalkDto })
  @ApiCreatedResponse({ type: ResponseDto })
  async chat(
    @Body(
      new ValidationPipe({
        transform: true,
        transformOptions: { enableImplicitConversion: true },
      }),
    )
    dto: TalkDto,
    @Headers('Authorization') header: string,
  ) {
    await this.authService.auth(header);
    return new ResponseDto(201, 'success', await this.chatgpt.chat(dto));
  }

  @Post('reset-conversation')
  @ApiOperation({ summary: 'Reset conversation' })
  @ApiBody({ type: SessionDto })
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  async resetConversation(
    @Body(
      new ValidationPipe({
        transform: true,
        transformOptions: { enableImplicitConversion: true },
      }),
    )
    dto: SessionDto,
    @Headers('Authorization') header: string,
  ) {
    await this.authService.auth(header);
    await this.conversation.resetConversation(dto.session);
    return new BlankReturnMessageDto(201, 'success');
  }

  @Get('has-conversation')
  @ApiOperation({ summary: 'Check where it has a conversation' })
  @ApiOkResponse({ type: hasConversationResponse })
  async hasConversation(
    @Query(
      new ValidationPipe({
        transform: true,
        transformOptions: { enableImplicitConversion: true },
      }),
    )
    dto: SessionDto,
    @Headers('Authorization') header: string,
  ) {
    await this.authService.auth(header);
    const hasConversation = await this.chatgpt.hasConversation(dto.session);
    const payload = new HasConversationDto();
    payload.session = dto.session;
    payload.hasConversation = hasConversation;
    return new hasConversationResponse(200, 'success', payload);
  }

  @Get('status')
  @ApiOperation({ summary: 'Get account status.' })
  @ApiOkResponse({ type: StatusDto })
  async status(@Headers('Authorization') header: string) {
    await this.authService.auth(header);
    return new StatusDto(200, 'success', this.accountPool.getAccountStatus());
  }
}
